package ru.yandex.chemodan.app.queller.celery.routing;

import org.eclipse.jetty.util.ConcurrentHashSet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.queller.celery.settings.task.CeleryTaskRegistry;
import ru.yandex.chemodan.queller.celery.job.CeleryTask;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.pg.fetcher.PgBazingaJobsFetcher;
import ru.yandex.commune.bazinga.pg.fetcher.TaskQueueConsumer;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * @author dbrylev
 * @author yashunsky
 */
public class CeleryBazingaJobsConsumer {
    private volatile MapF<String, QueueConsumer> consumers = Cf.map();

    private final PgBazingaJobsFetcher bazingaJobsFetcher;
    private final CeleryExecutionQueues celeryExecutionQueues;
    private final CeleryTasksDirector tasksDirector;

    private final DynamicProperty<Boolean> bazingaForcedUnavailable =
            new DynamicProperty<>("queller.bazinga.forced-unavailable.fetcher", false);

    public CeleryBazingaJobsConsumer(
            PgBazingaJobsFetcher bazingaJobsFetcher,
            CeleryExecutionQueues celeryExecutionQueues,
            CeleryTaskRegistry celeryTaskRegistry,
            CeleryTasksDirector tasksDirector)
    {
        this.bazingaJobsFetcher = bazingaJobsFetcher;
        this.celeryExecutionQueues = celeryExecutionQueues;
        this.tasksDirector = tasksDirector;

        celeryTaskRegistry.addListener(this::tasksChanged);
    }

    protected void tasksChanged(CollectionF<CeleryTask> tasks) {
        MapF<String, ListF<TaskId>> taskIdsByQueue = tasks.groupByMapValues(t -> t.executionQueue, t -> t.id);
        MapF<String, QueueConsumer> newConsumers = Cf.hashMap();

        consumers.keySet().minus(taskIdsByQueue.keySet())
                .forEach(q -> bazingaJobsFetcher.removeConsumer(consumers.getTs(q)));

        taskIdsByQueue.forEach((queue, taskIds) -> {
            QueueConsumer consumer;

            if (!consumers.containsKeyTs(queue)) {
                consumer = new QueueConsumer(queue);
                bazingaJobsFetcher.addConsumer(consumer);

            } else {
                consumer = consumers.getTs(queue);
            }
            newConsumers.put(queue, consumer);
            consumer.taskIds.filter(id -> !taskIds.containsTs(id)).forEach(consumer.taskIds::removeTs);
            consumer.taskIds.addAll(taskIds);
        });
        consumers = newConsumers;
    }

    private class QueueConsumer implements TaskQueueConsumer {
        private final String queueName;
        private final SetF<TaskId> taskIds = Cf.x(new ConcurrentHashSet<>());

        public QueueConsumer(String queueName) {
            this.queueName = queueName;
        }

        @Override
        public SetF<TaskId> getTaskIds() {
            return taskIds;
        }

        @Override
        public int getRemainCount() {
            if (bazingaForcedUnavailable.get()) {
                return 0;
            }
            return celeryExecutionQueues.getEnqueueFromGlobalLimitRemainCount(queueName);
        }

        @Override
        public ListF<Boolean> consume(ListF<OnetimeJob> jobs) {
            return tasksDirector.sendToCelery(jobs, "fromBazinga");
        }
    }
}
