package ru.yandex.chemodan.app.queller.celery.routing;

import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.Queue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Unit;
import ru.yandex.chemodan.app.queller.celery.settings.queue.CeleryQueueSettings;
import ru.yandex.chemodan.app.queller.celery.settings.queue.CeleryQueueSettingsRegistry;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.rabbit.QueueState;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RabbitQueues;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author dbrylev
 * @author yashunsky
 */
public class CeleryExecutionQueues {
    private static final Logger logger = LoggerFactory.getLogger(CeleryExecutionQueues.class);

    private final DirectExchange executeExchange;
    private final RabbitPool rabbitPool;

    private final CeleryQueueSettingsRegistry queueSettingsRegistry;

    private final CeleryMetrics celeryMetrics;

    private final MapF<String, Unit> queueDeclaration = Cf.concurrentHashMap();

    public CeleryExecutionQueues(
            DirectExchange executeExchange,
            RabbitPool rabbitPool,
            CeleryQueueSettingsRegistry queueSettingsRegistry,
            CeleryMetrics celeryMetrics)
    {
        this.rabbitPool = rabbitPool;
        this.executeExchange = executeExchange;
        this.queueSettingsRegistry = queueSettingsRegistry;
        this.celeryMetrics = celeryMetrics;
    }

    public QueueState getQueueState(String queueName) {
        queueDeclaration.computeIfAbsent(queueName, name -> {
            Queue queue = RabbitQueues.durable(name);

            rabbitPool.declareQueue(queue);
            rabbitPool.declareBinding(BindingBuilder.bind(queue).to(executeExchange).withQueueName());

            return Unit.U;
        });

        QueueState result = rabbitPool.aggregateQueueStates(queueName);

        celeryMetrics.executionQueuesConsumerCount.set(result.consumerCount, new MetricName(queueName));
        celeryMetrics.executionQueuesMessageCount.set(result.messageCount, new MetricName(queueName));

        return result;
    }

    public int getEnqueueFromLocalLimitRemainCount(String queueName) {
        return remainCount(queueName, getSettings(queueName).enqueueFromLocalLengthLimit);
    }

    public int getEnqueueFromGlobalLimitRemainCount(String queueName) {
        return remainCount(queueName, getSettings(queueName).enqueueFromGlobalLengthLimit);
    }

    private int remainCount(String queueName, int limit) {
        QueueState state = getQueueState(queueName);

        return state.consumerCount > 0 ? Math.max(0, limit - state.messageCount) : 0;
    }

    public CeleryQueueSettings getSettings(String queueName) {
        return queueSettingsRegistry.getOrDefault(queueName);
    }
}
