package ru.yandex.chemodan.app.queller.celery.routing;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.NotImplementedException;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Seconds;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.core.Queue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.queller.celery.settings.task.CeleryTaskRegistry;
import ru.yandex.chemodan.log.LoggerProxies;
import ru.yandex.chemodan.queller.celery.QuellerQueues;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.job.CeleryJobContext;
import ru.yandex.chemodan.queller.celery.job.CeleryOnetimeJobConverter;
import ru.yandex.chemodan.queller.celery.job.CeleryTask;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.rabbit.NoWorkingRabbitsException;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RabbitQueues;
import ru.yandex.chemodan.queller.rabbit.RoutedMessage;
import ru.yandex.chemodan.queller.rabbit.SendResult;
import ru.yandex.chemodan.queller.support.BenderJsonMessageConverter;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.worker.BazingaHostPort;
import ru.yandex.commune.bazinga.pg.storage.JobRunInfo;
import ru.yandex.commune.bazinga.pg.storage.JobSaveResult;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsPartitionShardResolver;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.enums.EnumUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.MeasureInfo;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author dbrylev
 * @author yashunsky
 */
public class CeleryTasksDirector {

    public static final Tuple2List<Queue, Duration> PG_REJECTED_QUEUES =
            Cf.list(5, 10, 30, 60, 120, 300, 600, 1800, 3600, 7200, 21_600, 86_400, 0).toTuple2List(seconds -> {

                int min = seconds / 60, sec = seconds % 60;

                String suffix = seconds > 0 ? "-" + min + (sec > 0 ? String.format(".%02d", sec) : "") : "";

                return Tuple2.tuple(RabbitQueues.durable("pgRejected" + suffix), Duration.standardSeconds(seconds));
            });

    private static final Logger logger = LoggerFactory.getLogger(CeleryTasksDirector.class);

    private static final BenderJsonMessageConverter<CeleryJob> messageConverter =
            new BenderJsonMessageConverter<>(CeleryJob.class);

    private final RabbitPool rabbitPool;
    private final CeleryTaskRegistry taskRegistry;
    private final CeleryExecutionQueues executionQueues;
    private final PgBazingaStorage storage;
    private final JobsPartitionShardResolver jobsPartitioning;
    private final BazingaHostPort bazingaWorkerId;
    private final CeleryMetrics celeryMetrics;

    private final Duration sleepOnNoWorkingRabbits;
    private final Duration sleepOnProfitlessPgRejected;
    private final Duration sleepOnProfitlessStartCompleted;

    private final Duration startedCompletedHandlerTimeout;

    private final DynamicProperty<Boolean> bazingaForcedUnavailableSubmit =
            new DynamicProperty<>("queller.bazinga.forced-unavailable.submit", false);

    private final DynamicProperty<Boolean> bazingaForcedUnavailableCompletion =
            new DynamicProperty<>("queller.bazinga.forced-unavailable.completion", false);

    private final DynamicProperty<Integer> scheduleLocallyThreshold =
            new DynamicProperty<>("queller.schedule-locally.threshold-seconds", -1);

    public CeleryTasksDirector(
            RabbitPool rabbitPool,
            CeleryTaskRegistry taskRegistry,
            CeleryExecutionQueues executionQueues,
            PgBazingaStorage storage,
            JobsPartitionShardResolver jobsPartitioning,
            BazingaHostPort bazingaWorkerId,
            Duration sleepOnNoWorkingRabbits,
            Duration sleepOnProfitlessPgRejected,
            Duration sleepOnProfitlessStartCompleted,
            Duration startedCompletedHandlerTimeout,
            CeleryMetrics celeryMetrics)
    {
        this.rabbitPool = rabbitPool;
        this.taskRegistry = taskRegistry;
        this.executionQueues = executionQueues;
        this.storage = storage;
        this.jobsPartitioning = jobsPartitioning;
        this.bazingaWorkerId = bazingaWorkerId;
        this.sleepOnNoWorkingRabbits = sleepOnNoWorkingRabbits;
        this.sleepOnProfitlessPgRejected = sleepOnProfitlessPgRejected;
        this.sleepOnProfitlessStartCompleted = sleepOnProfitlessStartCompleted;
        this.startedCompletedHandlerTimeout = startedCompletedHandlerTimeout;
        this.celeryMetrics = celeryMetrics;
    }

    public ListF<Boolean> handleSubmit(ListF<CeleryJob> jobs, Option<Queue> rejectedQueue, boolean executeImmediately) {

        Tuple2List<RoutedJob, Integer> routedJobs = jobs.map(j -> routeJob(j, executeImmediately)).zipWithIndex();

        Tuple2List<RoutedJob, Integer> toBazinga = routedJobs.filterBy1(RoutedJob::routedToBazingaOrImmediateExecution);
        Tuple2List<RoutedJob, Integer> toCelery = routedJobs.filterBy1(RoutedJob::routedToCelery);

        Tuple2List<Boolean, Integer> bazingaAcks = scheduleToBazinga(toBazinga.get1(), rejectedQueue).zip(toBazinga.get2());
        Tuple2List<Boolean, Integer> celeryAcks = sendToCeleryExecution(toCelery.get1()).zip(toCelery.get2());

        SetF<Integer> acked = bazingaAcks.filterBy1(a -> a).get2().unique().plus(celeryAcks.filterBy1(a -> a).get2());

        if (rejectedQueue.isPresent() && jobs.isNotEmpty() && acked.isEmpty()) {
            ThreadUtils.sleep(sleepOnProfitlessPgRejected);
        }
        return routedJobs.get2().map(acked::containsTs);
    }

    private ListF<Boolean> sendToCeleryExecution(ListF<RoutedJob> jobs) {
        jobs = jobs.map(job -> job.withJob(new CeleryJob(
                job.job.task, job.job.id, job.job.retries, job.job.args, job.job.getKwargs(),
                job.job.utc, job.job.expires, Option.empty(),
                job.job.callbacks, job.job.errbacks, job.job.chord, job.job.taskset,
                job.task.hardTimeout.map(d -> Seconds.seconds((int) d.getStandardSeconds())),
                job.task.softTimeout.map(d -> Seconds.seconds((int) d.getStandardSeconds())),
                Option.of(job.job.getContext().getOrElse(CeleryJobContext.empty())
                        .withMaxRetries(job.task.getMaxRetires()).reducedForSubmit()))));

        return sendToCelery(jobs, rabbitPool::sendToMostConsumedRabbitsWithConfirmation);
    }

    private ListF<Boolean> sentToCeleryPgRejected(ListF<RoutedJob> jobs) {
        return sendToCelery(jobs, rabbitPool::sendToMostFreeRabbitsWithConfirmation);
    }

    private ListF<Boolean> sendToCelery(
            ListF<RoutedJob> jobs, Function<ListF<RoutedMessage>, ListF<SendResult>> sender)
    {
        ListF<SendResult> result;

        try {
            result = sender.apply(jobs.map(job -> new RoutedMessage(
                    messageConverter.toMessage(job.job, new MessageProperties()),
                    QuellerQueues.EXECUTE_EXCHANGE.getName(), Option.of(job.task.executionQueue))));

        } catch (NoWorkingRabbitsException t) {
            logger.error("Tasks resubmit failed: {}", t);
            ThreadUtils.sleep(sleepOnNoWorkingRabbits);
            throw t;
        }

        IteratorF<SendResult> resultIt = result.iterator();

        jobs.forEach(rj -> {
            SendResult res = resultIt.next();

            if (res.isSentConfirmed()) {
                celeryMetrics.tasks.inc(new MetricName("sentToCelery", rj.comment, rj.job.task.toString()));
                ycridLogger(rj).info("Task {} resubmitted into {}", rj.forLog(), rj.task.executionQueue);
            } else {
                ycridLogger(rj).warn("Task {} will be processed again with sending status {}", rj.forLog(), res.status);
            }

            celeryMetrics.tasksSending.inc(MetricName.EMPTY);
            celeryMetrics.tasksSending.inc(EnumUtils.toXmlName(res.status));
        });

        return result.map(SendResult::isSentConfirmed);
    }

    private ListF<Boolean> scheduleToBazinga(ListF<RoutedJob> jobs, Option<Queue> rejectedQueue) {
        ListF<BazingaReaction> results = jobs.map(rj -> {
            if (bazingaForcedUnavailableSubmit.get()) {
                return new BazingaReaction.SendToRejected(chooseRejectedQueue(rj));
            }
            if (rj.comment.equals(RouteComments.SCHEDULED_LOCALLY)) {
                return new BazingaReaction.SendToRejected(chooseRejectedQueue(rj));
            }
            try {
                Option<CeleryJob> submitToExecute = Option.empty();

                if (rj.comment.equals(RouteComments.FAILED)) {
                    failJob(rj.task, rj.job);

                } else if (rj.routedToImmediateExecution()) {
                    submitToExecute = startAtBazingaIfAbsent(rj);

                    if (submitToExecute.isPresent()) {
                        ycridLogger(rj).info("Task {} added as starting", rj.forLog());
                    } else {
                        ycridLogger(rj).warn("Task {} was not added as starting", rj.forLog());
                    }
                } else {
                    FullJobId savedId = saveToBazinga(rj.task, rj.job, JobStatus.READY, Option.of(rj.scheduleTime));

                    if (savedId.equals(CeleryOnetimeJobConverter.getJobId(rj.job))) {
                        ycridLogger(rj).info("Task {} saved to be executed at {}", rj.forLog(), rj.scheduleTime);
                    } else {
                        ycridLogger(rj).info("Task {} merged by active uid with {}", rj.forLog(), savedId);
                    }
                }

                celeryMetrics.tasks.inc(new MetricName("sentToBazinga", rj.comment, rj.job.task.toString()));
                return submitToExecute.isPresent()
                        ? new BazingaReaction.SendToExecutionOrRestoreReady(rj.withJob(submitToExecute.get()))
                        : new BazingaReaction.Done();

            } catch (Throwable t) {
                ExceptionUtils.throwIfUnrecoverable(t);

                if (rj.comment.equals(RouteComments.FULL_QUEUES) && rejectedQueue.isSome(PG_REJECTED_QUEUES.first().get1())) {
                    ycridLogger(rj).warn("Task {} will be submitted to celery despite bazinga error: {}",
                            rj.forLog(), ExceptionUtils.getAllMessages(t));

                    return new BazingaReaction.SendToExecution(rj);
                }

                ycridLogger(rj).warn("Task {} will be processed again because of bazinga error: {}",
                        rj.forLog(), ExceptionUtils.getAllMessages(t));

                return new BazingaReaction.SendToRejected(chooseRejectedQueue(rj));
            }
        });

        IteratorF<Boolean> executions = sendToCeleryExecution(results.filterMap(
                r -> r.asSendToExecution().map(e -> e.job))).iterator();

        IteratorF<Boolean> moves = sentToCeleryPgRejected(results.zip(jobs).filterMap(
                t -> t.get1().asMoveToAnotherRejected(rejectedQueue).map(q -> t.get2().withQueueName(q.queue.getName())))
        ).iterator();

        return results.map(r -> {
            if (r.isSendToExecutionOrRestoreReady()) {
                return executions.next() || updateJobStartingToReadySafe(r.asSendToExecution().get().job);
            } else if (r.isSendToExecution()) {
                return executions.next();
            } else if (r.isMoveToAnotherRejected(rejectedQueue)) {
                return moves.next();
            } else {
                return r.isDone();
            }
        });
    }

    private Queue chooseRejectedQueue(RoutedJob job) {
        if (job.comment.equals(RouteComments.FAILED)) {
            return PG_REJECTED_QUEUES.last().get1();
        }

        Duration delay = new Duration(Instant.now(), job.scheduleTime);

        return PG_REJECTED_QUEUES.findBy2(delay::isShorterThan)
                .getOrElse(PG_REJECTED_QUEUES.last()).get1();
    }

    public ListF<Boolean> sendToCelery(ListF<OnetimeJob> jobs, String comment) {

        ListF<RoutedJob> routedJobs = jobs.map(
                job -> RoutedJob.toCelery(taskRegistry.get(job.getTaskId()),
                        CeleryOnetimeJobConverter.convertToCelery(job, true), comment)
        );

        return sendToCeleryExecution(routedJobs);
    }

    private RoutedJob routeJob(CeleryJob job, boolean executeImmediately) {
        CeleryJobContext context = job.getContext().getOrElse(CeleryJobContext.empty());

        CeleryTask task = taskRegistry.getOrRegisterAndGetTask(job);
        Instant now = Instant.now();

        Option<Instant> scheduleAt;

        if (!job.eta.isPresent()) {
            scheduleAt = job.retries > 0
                    ? task.reschedulePolicy.rescheduleAt(now, job.retries)
                    : Option.of(now);
        } else {
            scheduleAt = job.eta.map(DateTime::toInstant);
        }
        if (scheduleAt.isPresent()) {
            scheduleAt = scheduleAt.exists(now::isBefore) ? scheduleAt : Option.empty();

            if (job.retries > 0) {
                ycridLogger(job).warn("Task {} failed: {}. {}",
                        job.forLog(), context.error.getOrElse(""), context.traceback.getOrElse(""));

                celeryMetrics.tasksDone.update(
                        new MeasureInfo(new Duration(context.started.getOrElse(Instant.now()), Instant.now()), false),
                        Cf.list(new MetricName(job.task.toString()))
                );
            }
            int remainCount = executionQueues.getEnqueueFromLocalLimitRemainCount(task.executionQueue);

            boolean queueable = remainCount > 0;

            if (!scheduleAt.isPresent() && !queueable && !bazingaForcedUnavailableSubmit.get()) {
                if (executionQueues.getSettings(task.executionQueue).enqueueFromLocalLengthLimit > 0) {
                    ycridLogger(job).warn("Queue {} is full of tasks, saving task {} to be executed with bazinga",
                            task.executionQueue, job.forLog());

                    return RoutedJob.toBazinga(task, job, now, RouteComments.FULL_QUEUES);
                }
                if (executeImmediately && executionQueues.getEnqueueFromGlobalLimitRemainCount(task.executionQueue) > 0) {
                    return RoutedJob.toImmediateExecution(task, job, RouteComments.FULL_QUEUES);
                }
                return RoutedJob.toBazinga(task, job, now, RouteComments.FULL_QUEUES);

            } else if (!scheduleAt.isPresent()) {
                return RoutedJob.toCelery(task, job, queueable ? RouteComments.FROM_RABBIT_MQ : RouteComments.FULL_QUEUES);

            } else if (queueable && new Duration(now, scheduleAt.get()).getStandardSeconds() <= scheduleLocallyThreshold.get()) {
                return RoutedJob.toBazinga(task, job, scheduleAt.get(), RouteComments.SCHEDULED_LOCALLY);

            } else {
                return RoutedJob.toBazinga(task, job, scheduleAt.get(), RouteComments.SCHEDULED);
            }
        } else {
            return RoutedJob.toBazinga(task, job, now, RouteComments.FAILED);
        }
    }

    public ListF<Boolean> handleStartOrComplete(
            ListF<CeleryJob> jobs, Function<CeleryJob, Boolean> processor, Queue rejectQueue)
    {
        ListF<Boolean> statuses = jobs.map(
                job -> handleWithReject(job, processor, rejectQueue, startedCompletedHandlerTimeout));

        if (statuses.isNotEmpty() && !statuses.containsTs(true)) {
            ThreadUtils.sleep(sleepOnProfitlessStartCompleted);
        }
        return statuses;
    }

    private boolean handleWithReject(
            CeleryJob job, Function<CeleryJob, Boolean> processor, Queue rejectQueue, Duration timeout)
    {
        CompletableFuture<Boolean> processedJob = CompletableFuture.supplyAsync(() -> processor.apply(job));
        boolean handled;
        try {
            handled = processedJob.get(timeout.getMillis(), TimeUnit.MILLISECONDS);
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            handled = false;
        }

        if (handled) {
            return true;
        }
        ycridLogger(job).warn("Failed to process task {} within {}. Resubmitting to {}",
                job.forLog(), timeout, rejectQueue.getName());
        return submitToRejectQueue(job, rejectQueue);
    }

    private boolean submitToRejectQueue(CeleryJob job, Queue rejectQueue) {
        try {
            RoutedMessage routedMessage = new RoutedMessage(
                    messageConverter.toMessage(job, new MessageProperties()),
                    QuellerQueues.SUBMIT_EXCHANGE.getName(), Option.of(rejectQueue.getName()));

            return rabbitPool.sendToMostFreeRabbitsWithConfirmation(Cf.list(routedMessage)).exists(SendResult::isSent);
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            ycridLogger(job).warn("Failed submit task {} to {}", job.forLog(), rejectQueue.getName());
            return false;
        }
    }

    public boolean handleStarted(CeleryJob job) {
        CeleryJobContext context = job.getContext().getOrElse(CeleryJobContext.empty());

        if (context.globalQueued) {
            JobRunInfo info = new JobRunInfo(
                    JobStatus.RUNNING, Option.empty(),
                    context.started.orElse(Option.of(Instant.now())), Option.empty());

            CeleryTask task = taskRegistry.getOrRegisterAndGetTask(job);

            if (!startOrCompleteJob(job, () -> updateJobStartingToRunning(task, job, info))) {
                return false;
            }
        }
        ycridLogger(job).info("Task {} was ran by {} at {}",
                job.forLog(), context.worker.getOrNull(), context.started.mkString(""));

        celeryMetrics.tasks.inc(new MetricName("ran", job.task.toString()));
        return true;
    }

    public boolean handleCompleted(CeleryJob job) {
        CeleryJobContext context = job.getContext().getOrElse(CeleryJobContext.empty());

        if (context.globalQueued) {
            CeleryTask task = taskRegistry.getOrRegisterAndGetTask(job);

            if (!startOrCompleteJob(job, () -> saveToBazinga(task, job, JobStatus.COMPLETED, Option.empty()))) {
                return false;
            }
        }

        Instant finished = context.finished.getOrElse(Instant.now());
        Instant started = context.started.getOrElse(finished);
        Duration duration = new Duration(started, finished);

        ycridLogger(job).info("Task {} completed successfully by {}, took {}",
                job.forLog(), context.worker.getOrNull(),
                TimeUtils.millisecondsToSecondsString(duration.getMillis()));

        celeryMetrics.tasksDone.update(
                new MeasureInfo(duration, true),
                Cf.list(new MetricName(job.task.toString()))
        );
        return true;
    }

    private boolean startOrCompleteJob(CeleryJob job, Function0<?> action) {
        if (bazingaForcedUnavailableCompletion.get()) {
            return false;
        }
        return updateJobStatusSafe(job, action);
    }

    private boolean updateJobStatusSafe(CeleryJob job, Function0<?> action) {
        try {
            action.apply();
            return true;

        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);

            ycridLogger(job).warn("Failed to update {} task status: {}", job.forLog(), t);
            return false;
        }
    }

    private void failJob(CeleryTask task, CeleryJob job) {
        CeleryJobContext context = job.getContext().getOrElse(CeleryJobContext.empty());

        if (context.globalQueued) {
            saveToBazinga(task, job, JobStatus.FAILED, Option.empty());
        }

        ycridLogger(job).error("Task {} failed: {}. {}",
                job.forLog(), context.error.getOrElse(""), context.traceback.getOrElse(""));

        celeryMetrics.tasksDone.update(
                new MeasureInfo(new Duration(context.started.getOrElse(Instant.now()), Instant.now()), false),
                Cf.list(new MetricName(job.task.toString()))
        );

        celeryMetrics.tasks.inc(new MetricName("failed", job.task.toString()));
    }

    private boolean updateJobStartingToReadySafe(RoutedJob job) {
        return updateJobStatusSafe(job.job, () -> updateStartingJobRunInfo(
                job.task, job.job, JobRunInfo.ready(), Option.of(job.scheduleTime)));
    }

    private boolean updateJobStartingToRunning(CeleryTask task, CeleryJob job, JobRunInfo info) {
        Validate.equals(info.status, JobStatus.RUNNING);
        return updateStartingJobRunInfo(task, job, info, Option.empty());
    }

    private boolean updateStartingJobRunInfo(CeleryTask task, CeleryJob job, JobRunInfo info, Option<Instant> scheduleTime) {
        OnetimeJob onetimeJob = CeleryOnetimeJobConverter.convertFromCelery(
                task, job, info, scheduleTime, jobsPartitioning);

        return storage.updateOnetimeJobRunInfoByIdAndStatus(onetimeJob, JobStatus.STARTING);
    }

    private Option<CeleryJob> startAtBazingaIfAbsent(RoutedJob job) {
        if (job.job.getContext().exists(c -> c.activeUid.isPresent())
                && job.task.activeUidBehavior.getDuplicateBehavior() != ActiveUidDuplicateBehavior.DO_NOTHING)
        {
            throw new NotImplementedException("Hard to deal with merge by active uid");
        }
        JobRunInfo runInfo = new JobRunInfo(
                JobStatus.STARTING, Option.of(bazingaWorkerId),
                Option.of(job.scheduleTime), Option.empty());

        OnetimeJob onetimeJob = CeleryOnetimeJobConverter.convertFromCelery(
                job.task, job.job, runInfo, Option.of(job.scheduleTime), jobsPartitioning);

        JobSaveResult result = storage.addOnetimeJobIfAbsent(onetimeJob, job.task.activeUidBehavior.getDuplicateBehavior());

        return Option.when(result.isCreated(), CeleryOnetimeJobConverter.convertToCelery(onetimeJob, true));
    }

    private FullJobId saveToBazinga(CeleryTask task, CeleryJob job, JobStatus status, Option<Instant> scheduleTime) {
        OnetimeJob onetimeJob = CeleryOnetimeJobConverter.convertFromCelery(task, job, status, scheduleTime, jobsPartitioning);

        if (job.isGlobalQueued()) {
            return storage.saveOnetimeJob(onetimeJob, task.activeUidBehavior.getDuplicateBehavior());
        } else {
            return storage.addOnetimeJobIfAbsent(onetimeJob, task.activeUidBehavior.getDuplicateBehavior()).getJobId();
        }
    }

    private Logger ycridLogger(RoutedJob job) {
        return ycridLogger(job.job);
    }

    private Logger ycridLogger(CeleryJob job) {
        return LoggerProxies.withYcrid(logger, job.getYcrid());
    }
}
