package ru.yandex.chemodan.app.queller.celery.routing;

import org.joda.time.Instant;

import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.job.CeleryTask;

/**
 * @author yashunsky
 */
public class RoutedJob {

    private enum Destination {
        CELERY,
        BAZINGA,
        IMMEDIATE_EXECUTION,
    }

    private final Destination sendTo;
    public final CeleryTask task;
    public final CeleryJob job;
    public final Instant scheduleTime;
    public final String comment;

    public static RoutedJob toCelery(CeleryTask task, CeleryJob job, String comment) {
        return new RoutedJob(Destination.CELERY, task, job, Instant.now(), comment);
    }

    public static RoutedJob toBazinga(CeleryTask task, CeleryJob job, Instant scheduleTime, String comment) {
        return new RoutedJob(Destination.BAZINGA, task, job, scheduleTime, comment);
    }

    public static RoutedJob toImmediateExecution(CeleryTask task, CeleryJob job, String comment) {
        return new RoutedJob(Destination.IMMEDIATE_EXECUTION, task, job, Instant.now(), comment);
    }

    private RoutedJob(Destination sendTo, CeleryTask task, CeleryJob job, Instant scheduleTime, String comment)
    {
        this.sendTo = sendTo;
        this.task = task;
        this.job = job;
        this.scheduleTime = scheduleTime;
        this.comment = comment;
    }

    public boolean routedToCelery() {
        return sendTo == Destination.CELERY;
    }

    public boolean routedToBazingaOrImmediateExecution() {
        return sendTo == Destination.BAZINGA || sendTo == Destination.IMMEDIATE_EXECUTION;
    }

    public boolean routedToImmediateExecution() {
        return sendTo == Destination.IMMEDIATE_EXECUTION;
    }

    public String forLog() {
        return job.forLog();
    }

    public RoutedJob withQueueName(String queueName) {
        return new RoutedJob(Destination.CELERY, task.withQueueName(queueName), job, scheduleTime, comment);
    }

    public RoutedJob withJob(CeleryJob job) {
        return new RoutedJob(sendTo, task, job, scheduleTime, comment);
    }
}
