package ru.yandex.chemodan.app.queller.celery.settings.monitor;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function3V;
import ru.yandex.chemodan.app.queller.celery.settings.worker.WorkerAssignmentRulesManager;
import ru.yandex.chemodan.queller.celery.control.CeleryControl;
import ru.yandex.chemodan.queller.celery.control.callback.CeleryInspectActiveCallback;
import ru.yandex.chemodan.queller.celery.control.callback.CeleryInspectActiveQueuesCallback;
import ru.yandex.chemodan.queller.celery.control.callback.CeleryInspectStatsCallback;
import ru.yandex.chemodan.queller.celery.control.callback.CeleryReplyInfo;
import ru.yandex.chemodan.queller.celery.control.callback.JavaWorkerInspectStatsCallback;
import ru.yandex.chemodan.queller.celery.control.callback.replies.CeleryInspectActiveQueuesReply;
import ru.yandex.chemodan.queller.celery.control.callback.replies.CeleryInspectActiveReply;
import ru.yandex.chemodan.queller.celery.control.callback.replies.CeleryInspectStatsReply;
import ru.yandex.chemodan.queller.celery.control.callback.replies.JavaWorkerInspectStatsReply;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.celery.worker.WorkerState;
import ru.yandex.chemodan.queller.celery.worker.WorkerStateProvider;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.MeasureInfo;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */
public class CeleryMonitor extends DelayingWorkerServiceBeanSupport implements WorkerStateProvider {
    private static final Logger logger = LoggerFactory.getLogger(CeleryMonitor.class);

    private final DynamicProperty<Integer> workerTimeout =
            new DynamicProperty<>("celery-monitor.worker-timeout", 10000);

    private final DynamicProperty<Integer> balancingUnchangedChecks =
            new DynamicProperty<>("celery-monitor.workers-balancing.unchanged-checks", 4);

    private final CeleryControl celery;

    private final RabbitPool rabbits;

    private final WorkerAssignmentRulesManager workerManager;

    private final CeleryMetrics celeryMetrics;

    private MapF<WorkerId, WorkerState> workersState;

    @Override
    public MapF<WorkerId, WorkerState> getWorkersState() {
        return workersState;
    }

    protected WorkerState getStatisticItemForUpdate(WorkerId worker) {
        //return an existing item from statistics with an updated timestamp
        //or create a new item and return it.
        WorkerState item;
        if (workersState.containsKeyTs(worker)) {
            item = workersState.getTs(worker);
        } else {
            item = new WorkerState();
            item.workerId = worker.toString();
            workersState.put(worker, item);
        }
        item.lastAnswer = Instant.now();
        return item;
    }

    protected boolean defaultSleepBeforeFirstRun() {
        return false;
    }

    private void updateCeleryReplyMetric(CeleryControl.Method method, WorkerId worker, Duration duration) {
        celeryMetrics.replies.update(new MeasureInfo(duration, true),
                Cf.list(
                        new MetricName(
                                CamelWords.parse(method.name()).toJavaIdentifier(),
                                worker.toString()
                        )
                )
        );
    }

    public CeleryMonitor(
            RabbitPool rabbitPool,
            CeleryControl celeryControl,
            WorkerAssignmentRulesManager workerAssignmentRulesManager,
            Duration maintenancePeriod,
            CeleryMetrics metrics)
    {
        celery = celeryControl;
        rabbits = rabbitPool;

        workerManager = workerAssignmentRulesManager;
        celeryMetrics = metrics;

        workersState = Cf.concurrentHashMap();

        setSleepBeforeFirstRun(false);
        setDelay(maintenancePeriod);

        celery.registerCallback(
                new CeleryInspectStatsCallback() {
                    public void onMessageGet(CeleryReplyInfo info, CeleryInspectStatsReply reply) {
                        if (reply.pool.targetProcesses.isPresent() && reply.pool.isConsuming.isPresent()) {
                            WorkerId worker = info.workerId;

                            // extract data from reply and store it
                            int poolSize = reply.pool.targetProcesses.get();

                            WorkerState item = getStatisticItemForUpdate(worker);
                            item.rabbitHosts = Option.of(reply.broker.hostname);
                            item.poolSize = reply.pool.targetProcesses;
                            item.processesCount = Option.of(reply.pool.processes.length());
                            item.isConsuming = reply.pool.isConsuming;

                            // get expected processes count from workers manager
                            int targetPoolSize = workerManager.getQueuesPool(worker).poolSize;

                            // adjust worker if required
                            if (poolSize != targetPoolSize) {
                                celery.controlSetPool(Option.of(Cf.list(worker)), targetPoolSize);

                                celeryMetrics.requests.inc(new MetricName("control", "setPool", worker.toString()));
                            }

                            //update metrics
                            celeryMetrics.isConsuming
                                    .set(reply.pool.isConsuming.get(), new MetricName(worker.toString()));

                            updateCeleryReplyMetric(this.method, worker, info.duration);
                        }
                    }
                }
        );

        celery.registerCallback(
                new CeleryInspectActiveCallback() {
                    public void onMessageGet(CeleryReplyInfo info, ListF<CeleryInspectActiveReply> replies) {
                        WorkerId worker = info.workerId;

                        // extract data from reply and store it
                        getStatisticItemForUpdate(worker).activeProcessesCount = Option.of(replies.length());

                        //update metrics
                        updateCeleryReplyMetric(this.method, worker, info.duration);
                    }
                }
        );

        celery.registerCallback(
                new CeleryInspectActiveQueuesCallback() {
                    public void onMessageGet(CeleryReplyInfo info, ListF<CeleryInspectActiveQueuesReply> replies) {
                        WorkerId worker = info.workerId;

                        // extract data from reply and store it
                        ListF<String> queues = replies.map(e -> e.name);
                        getStatisticItemForUpdate(worker).queues = queues;

                        SetF<String> currentQueues = Cf.toHashSet(queues);

                        // get expected queues names
                        SetF<String> requiredQueues = Cf.toHashSet(workerManager.getQueuesPool(worker).queues);

                        SetF<String> toCancel = currentQueues.minus(requiredQueues);
                        SetF<String> toAdd = requiredQueues.minus(currentQueues);

                        // adjust worker if required
                        Option<ListF<WorkerId>> destination = Option.of(Cf.list(worker));
                        toCancel.forEach(queue -> {
                            celery.controlCancelConsumer(destination, queue);
                            celeryMetrics.requests.inc(new MetricName("control", "cancelConsumer", worker.toString()));
                        });
                        toAdd.forEach(queue -> {
                            celery.controlAddConsumer(destination, queue, queue, "direct", queue);
                            celeryMetrics.requests.inc(new MetricName("control", "addConsumer", worker.toString()));
                        });

                        updateCeleryReplyMetric(this.method, worker, info.duration);
                    }
                }
        );

        celery.registerCallback(
                new JavaWorkerInspectStatsCallback() {
                    public void onMessageGet(CeleryReplyInfo info, JavaWorkerInspectStatsReply reply) {
                        WorkerState item = getStatisticItemForUpdate(info.workerId);

                        item.isJavaWorker = true;
                        item.rabbitHosts = reply.rabbitHosts;
                        item.queues = reply.queues;
                        item.isConsuming = Option.of(true);

                        updateCeleryReplyMetric(this.method, info.workerId, info.duration);
                    }
                }
        );
    }

    protected void cleanupWorkersList() {
        //Delete workers from workersState if the don't answer within allowed period.
        Instant limit = Instant.now().minus(workerTimeout.get());

        workersState.entrySet().removeIf(e -> e.getValue().lastAnswer.isBefore(limit));

        logger.info("Connected workers {}", workersState.keys().sorted(WorkerId.comparator));
    }

    private int balancingChecksCount = 0;
    private RabbitsWorkers balancingLastState = new RabbitsWorkers(Cf.list(), Cf.list());

    private void balanceWorkers() {
        RabbitsWorkers state = RabbitsWorkers.cons(rabbits.getWorkingRabbits(), workersState.values());

        if (balancingUnchangedChecks.get() > 0 && !balancingLastState.equals(state)) {
            balancingChecksCount = 0;
            balancingLastState = state;

        } else if (++balancingChecksCount >= balancingUnchangedChecks.get()) {
            Tuple2List<WorkerId, Host> changes = state.balancingAssignments();

            if (changes.isNotEmpty()) {
                logger.info("Resetting workers connections: {}",
                        changes.sortedBy1(WorkerId.comparator).mkString(", ", " to "));

                changes.forEach(c -> celery.controlResetConnection(Option.of(Cf.list(c.get1())), c.get2()));
            }
            balancingChecksCount = 0;
            balancingLastState = new RabbitsWorkers(Cf.list(), Cf.list());
        }
    }

    @Override
    protected void execute() throws Exception {
        cleanupWorkersList();
        // ensure input queue is ready;
        celery.declareInputQueueAndBinding();
        // Request statistics from all workers
        celery.inspectStats(Option.empty());
        celery.inspectActive(Option.empty());
        celery.inspectActiveQueues(Option.empty());

        ListF<CeleryControl.Method> methods = Cf.list(
                CeleryControl.Method.INSPECT_STATS,
                CeleryControl.Method.INSPECT_ACTIVE,
                CeleryControl.Method.INSPECT_ACTIVE_QUEUES
        );

        celeryMetrics.requests.inc(
                methods.map(method -> new MetricName(
                        "inspect",
                        CamelWords.parse(method.name()).toJavaIdentifier(),
                        "broadcast"
                ))
        );

        updateAggregatedWorkersStatusForYasm();

        balanceWorkers();
    }

    protected void updateAggregatedWorkersStatusForYasm() {

        Function3V<String, String, Integer> publishAggregated = (workerName, metricName, value) ->
                celeryMetrics.processes.set(value, new MetricName("aggregated", workerName, metricName));

        workersState.entries().map1(workerId -> workerId.name.name).map2(WorkerState::aggregated)
                .groupBy1().mapValues(states -> states.sum(WorkerState.Aggregated.ADDER))
                .forEach((workerName, state) -> {
                    publishAggregated.apply(workerName, "target", state.getPoolSize());
                    publishAggregated.apply(workerName, "real", state.getProcessesCount());
                    publishAggregated.apply(workerName, "active", state.getActiveProcessesCount());
                });

    }
}
