package ru.yandex.chemodan.app.queller.celery.settings.monitor;

import java.util.Comparator;
import java.util.PriorityQueue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.celery.worker.WorkerState;
import ru.yandex.chemodan.queller.rabbit.RabbitConnection;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author dbrylev
 */
public class RabbitsWorkers extends DefaultObject {

    public final ListF<Host> rabbits;
    public final ListF<RabbitWorker> rabbitsWorkers;

    public RabbitsWorkers(ListF<Host> rabbits, ListF<RabbitWorker> rabbitsWorkers) {
        this.rabbits = rabbits;
        this.rabbitsWorkers = rabbitsWorkers;
    }

    public static RabbitsWorkers cons(CollectionF<RabbitConnection> rabbits, CollectionF<WorkerState> workers) {
        return new RabbitsWorkers(
                rabbits.map(rc -> rc.connectionData.host),
                workers.filter(w -> !w.isJavaWorker && w.rabbitHosts.isNotEmpty()).map(RabbitWorker::cons));
    }

    public Tuple2List<WorkerId, Host> balancingAssignments() {
        Tuple2List<WorkerId, Host> result = Tuple2List.arrayList();

        for (ListF<RabbitWorker> workers: rabbitsWorkers.groupBy(w -> w.worker.name).values()) {
            int required = (int) Math.ceil((float) workers.size() / rabbits.size() / 2);

            MapF<Host, ListF<WorkerId>> workersByRabbit = workers.groupByMapValues(w -> w.rabbit, w -> w.worker);

            ListF<RabbitWorkers> rabbitsWorkers =
                    rabbits.map(h -> new RabbitWorkers(h, workersByRabbit.getOrElse(h, Cf.list())));

            Comparator<RabbitWorkers> comparator =
                    Comparator.comparing(RabbitWorkers::count).thenComparing(w -> w.rabbit.format());

            PriorityQueue<RabbitWorkers> overfed = new PriorityQueue<>(comparator.reversed());
            PriorityQueue<RabbitWorkers> underfed = new PriorityQueue<>(comparator);

            overfed.addAll(rabbitsWorkers.filter(r -> r.count() > required));
            underfed.addAll(rabbitsWorkers.filter(r -> r.count() < required));

            while (!overfed.isEmpty() && !underfed.isEmpty()) {
                RabbitWorkers source = overfed.poll();
                RabbitWorkers target = underfed.poll();

                target.workers.add(source.workers.remove(source.workers.size() - 1));

                overfed.addAll(Option.when(source.count() > required, source));
                underfed.addAll(Option.when(target.count() < required, target));

                result.add(target.workers.last(), target.rabbit);
            }
        }
        return result;
    }

    private static class RabbitWorkers extends DefaultObject {
        private final Host rabbit;
        private final ArrayListF<WorkerId> workers;

        public RabbitWorkers(Host rabbit, ListF<WorkerId> workers) {
            this.rabbit = rabbit;
            this.workers = new ArrayListF<>(workers.sorted(WorkerId.comparator));
        }

        public int count() {
            return workers.size();
        }
    }

    public static class RabbitWorker extends DefaultObject {
        public final Host rabbit;
        public final WorkerId worker;

        public RabbitWorker(Host rabbit, WorkerId worker) {
            this.rabbit = rabbit;
            this.worker = worker;
        }

        public static RabbitWorker cons(WorkerState state) {
            return new RabbitWorker(Host.parse(state.rabbitHosts.single()), WorkerId.parse(state.workerId));
        }
    }
}
