package ru.yandex.chemodan.app.queller.celery.settings.task;

import org.joda.time.Duration;
import org.joda.time.Seconds;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.job.CeleryTask;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;

/**
 * @author dbrylev
 */
public class CeleryTaskRegistry extends ZkRegistry<TaskId, CeleryTask> {

    private static final TaskId DEFAULTS_TASK_ID = new TaskId("defaults");
    private static final String DEFAULT_EXECUTION_QUEUE = "tasks-default";
    private static final int DEFAULT_PRIORITY = 0;
    private static final ActiveUidBehavior DEFAULT_ACTIVE_UID_BEHAVIOR = new ActiveUidBehavior(
            ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.MERGE);

    public CeleryTaskRegistry(ZkPath path) {
        super(path, BazingaBender.mapper.createParserSerializer(CeleryTask.class), t -> t.id, TaskId::getId);
    }

    public CeleryTask getDefaults() {
        return getO(DEFAULTS_TASK_ID).getOrElse(() -> new CeleryTask(
                DEFAULTS_TASK_ID, DEFAULT_EXECUTION_QUEUE, DEFAULT_PRIORITY,
                Option.of(Duration.standardMinutes(5)),
                Option.of(Duration.standardMinutes(5)),
                new CompoundReschedulePolicy(
                        new RescheduleConstant(Duration.ZERO, 3),
                        new RescheduleExponential(Duration.standardMinutes(2), 10)),
                DEFAULT_ACTIVE_UID_BEHAVIOR));
    }

    public CeleryTask getOrRegisterAndGetTask(CeleryJob job) {
        Option<CeleryTask> found = getO(job.task);

        if (found.isPresent()) return found.get();

        CeleryTask defaults = getDefaults();

        CeleryTask result = new CeleryTask(
                job.task, defaults.executionQueue, defaults.priority,
                job.getSoftTimeLimit().map(Seconds::toStandardDuration).orElse(defaults.softTimeout),
                job.getHardTimeLimit().map(Seconds::toStandardDuration).orElse(defaults.hardTimeout),
                defaults.reschedulePolicy, defaults.activeUidBehavior);

        putIfAbsent(result);

        return get(job.task);
    }

    public ListF<CeleryTask> getAllNonDefaults() {
        return getAll().iterator().filter(t -> !t.id.equals(DEFAULTS_TASK_ID)).toList();
    }
}
