package ru.yandex.chemodan.app.queller.celery.settings.worker;

import java.util.concurrent.CountDownLatch;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.queller.celery.settings.worker.rules.DeniedQueuesRules;
import ru.yandex.chemodan.app.queller.celery.settings.worker.rules.QueuesPoolRule;
import ru.yandex.chemodan.app.queller.celery.settings.worker.rules.RuleScope;
import ru.yandex.chemodan.app.queller.celery.settings.worker.rules.WorkerAssignmentRules;
import ru.yandex.chemodan.app.queller.celery.settings.worker.rules.WorkerQueuesPoolRules;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.celery.worker.WorkerName;
import ru.yandex.commune.alive2.location.Location;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.zk2.ZkClient;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.ZkWrapper;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.lang.Check;

/**
 * @author dbrylev
 */
public class WorkerAssignmentRulesManager implements ZkClient {
    private static final String ZK_NODE_NAME = "-";

    private final ZkRegistry<WorkerName, WorkerQueuesPoolRules> workerRulesRegistry;
    private final ZkRegistry<String, DeniedQueuesRules> deniedQueuesRegistry;

    private final CountDownLatch workerInitializedLatch = new CountDownLatch(1);
    private final CountDownLatch queuesInitializedLatch = new CountDownLatch(1);

    private volatile MapF<WorkerName, ListF<QueuesPoolRule>> orderedRulesByWorkerName;
    private volatile MapF<String, ListF<RuleScope>> denyScopesByQueueName;

    private final LocationResolver locationResolver;

    public WorkerAssignmentRulesManager(ZkPath workerRulesPath, ZkPath deniedQueuesPath, LocationResolver locationResolver) {
        this.workerRulesRegistry = new ZkRegistry<>(workerRulesPath,
                Bender.cons(WorkerQueuesPoolRules.class),
                e -> e.worker, w -> w.name);

        this.deniedQueuesRegistry = new ZkRegistry<>(deniedQueuesPath,
                Bender.cons(DeniedQueuesRules.class),
                e -> ZK_NODE_NAME, s -> s);

        this.locationResolver = locationResolver;
        this.workerRulesRegistry.addListener(this::workerRulesChanged);
        this.deniedQueuesRegistry.addListener(this::deniedQueuesChanged);
    }

    public WorkerAssignmentRules getRules() {
        return new WorkerAssignmentRules(workerRulesRegistry.getAll().toList(), getDeniedQueuesRules().rules);
    }

    public DeniedQueuesRules getDeniedQueuesRules() {
        return deniedQueuesRegistry.getO(ZK_NODE_NAME).getOrElse(new DeniedQueuesRules(Cf.list()));
    }

    public WorkerQueuesPoolRules getWorkerRules(WorkerName name) {
        return workerRulesRegistry.getO(name).getOrElse(new WorkerQueuesPoolRules(name, Cf.list()));
    }

    public void saveRules(DeniedQueuesRules rules) {
        deniedQueuesRegistry.put(rules);
    }

    public void saveRules(WorkerQueuesPoolRules rules) {
        workerRulesRegistry.put(rules);
    }

    public void remove(WorkerName workerName) {
        workerRulesRegistry.remove(workerName);
    }

    public WorkerQueuesPool getQueuesPool(WorkerId workerId) {
        checkInitialized();

        Option<ListF<QueuesPoolRule>> orderedRulesO = orderedRulesByWorkerName.getO(workerId.name);

        if (!orderedRulesO.isPresent()) {
            return WorkerQueuesPool.empty();
        }

        Location location = locationResolver.resolveLocationFor(workerId.host.map(Host::format));

        WorkerQueuesPool queuesPool = orderedRulesO.get()
                .find(r -> r.scope.matches(location))
                .map(r -> r.queuesPool)
                .getOrElse(WorkerQueuesPool.empty());

        ListF<String> queues = queuesPool.queues
                .filter(q -> !denyScopesByQueueName.getOrElse(q, Cf.list()).exists(s -> s.matches(location)));

        return new WorkerQueuesPool(queues, queuesPool.poolSize);
    }

    private void checkInitialized() {
        Check.isTrue(workerInitializedLatch.getCount() == 0 && queuesInitializedLatch.getCount() == 0,
                "Not initialized");
    }

    private void workerRulesChanged(CollectionF<WorkerQueuesPoolRules> entries) {
        orderedRulesByWorkerName = entries
                .groupBy(r -> r.worker)
                .mapValues(rs -> rs.flatMap(r -> r.rules).reverse().sortedByDesc(QueuesPoolRule::priority));

        workerInitializedLatch.countDown();
    }

    private void deniedQueuesChanged(CollectionF<DeniedQueuesRules> entries) {
        DeniedQueuesRules rules = getDeniedQueuesRules();

        denyScopesByQueueName = rules.rules
                .flatMap(r -> r.queues.map(q -> Tuple2.tuple(q, r.scope)))
                .groupBy(Tuple2::get1)
                .mapValues(rs -> rs.map(Tuple2::get2));

        queuesInitializedLatch.countDown();
    }

    @Override
    public void connected(ZkWrapper zk) {
        workerRulesRegistry.connected(zk);
        deniedQueuesRegistry.connected(zk);
    }

    @Override
    public void expired() {
        workerRulesRegistry.expired();
        deniedQueuesRegistry.expired();
    }
}
