package ru.yandex.chemodan.app.smartcache.client.actions;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.PhotoFieldsPojo;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.ResponsePojoMappers;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.SnapshotAlbumPojo;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.SnapshotClusterItemPojo;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.SnapshotResponsePojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizationType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestsContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.utils.PhotoViewCacheBender;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
@ContextConfiguration(classes = {
        TestsContextConfiguration.class
})
@TestExecutionListeners(value = {
        DependencyInjectionTestExecutionListener.class
})
@ActivateDataApiEmbeddedPg
public class AlbumsSerializationTest extends DataApiTestSupport {

    @Autowired
    private DataApiStorageManager dataApiStorageManager;

    private DataApiUserId testUid;

    @Before
    public void before() {
        testUid = createRandomCleanUserInDefaultShard();
    }

    @Test
    public void serializationTest() {
        PhotoFieldsPojo pojo = new PhotoFieldsPojo("/disk/photo.jpg", Option.empty(), Option.empty(), Option.of(10.0),
                Option.of(Cf.set(AlbumType.BEAUTIFUL)));
        String json = new String(PhotoViewCacheBender.clientA3Mapper().serializeJson(pojo));
        Assert.equals("{\"path\":\"/disk/photo.jpg\",\"beauty\":10.0,\"albums\":[\"beautiful\"]}", json);
    }

    @Test
    public void testClusterWithAlbums() {
        testResponsePojo(true);
    }

    @Test
    public void testClusterWithNoAlbums() {
        testResponsePojo(false);
    }

    private void testResponsePojo(boolean withAlbums) {

        PhotoViewLuceneClusterPojo oldLuceneCluster = new PhotoViewLuceneClusterPojo(2,
                new Instant(1417268610), new Instant(1417324906),
                Cf.list(createRandomPhotoWithBeauty("old beauty", 10), createRandomPhotoWithBeauty("old unbeauty", -10)));

        PhotoViewLuceneClusterPojo newLuceneCluster = new PhotoViewLuceneClusterPojo(2,
                new Instant(1417325906), new Instant(1417325906),
                Cf.list(createRandomPhotoWithBeauty("new beauty", 10)));

        if (!withAlbums) {
            oldLuceneCluster = oldLuceneCluster.withNoAlbums();
            newLuceneCluster = newLuceneCluster.withNoAlbums();
        }

        IndexedCluster oldCluster = getIndexedCluster(oldLuceneCluster);
        IndexedCluster newCluster = getIndexedCluster(newLuceneCluster);

        ListF<RecordChange> recordChanges = IndexToDeltaMapper.insertClusterToSnapshot(oldCluster)
                .plus(IndexToDeltaMapper.insertClusterToSnapshot(newCluster))
                .plus(AlbumsToDeltaMapper.getAlbumsChanges(Cf.list(), Cf.list(oldLuceneCluster, newLuceneCluster)));

        Database indexDatabase = withAlbums
                ? dataApiStorageManager.createDatabaseWithAlbums(testUid)
                : dataApiStorageManager.createDatabase(testUid);
        dataApiStorageManager.storeSnapshotIndex(indexDatabase, RevisionCheckMode.WHOLE_DATABASE, recordChanges);

        Database database = dataApiStorageManager.getDatabaseO(testUid).get();
        Snapshot snapshot = getSnapshot(database, Option.empty());

        DataFields indexDataFields = snapshot.records()
                .filter(record -> record.getCollectionId().equals(IndexToDeltaMapper.INDEX_COLLECTION_ID))
                .toList().last().getFields();

        if (withAlbums) {
            Assert.some(DataField.integer(1), indexDataFields.getO("albumBeautiful"));
            Assert.some(DataField.integer(1), indexDataFields.getO("albumUnbeautiful"));
            Assert.some(DataField.integer(2), indexDataFields.getO("albumNonphotounlim"));
        } else {
            Assert.isEmpty(indexDataFields.getO("albumBeautiful"));
            Assert.isEmpty(indexDataFields.getO("albumUnbeautiful"));
            Assert.isEmpty(indexDataFields.getO("albumNonphotounlim"));
        }

        SnapshotResponsePojo pojo = ResponsePojoMappers.snapshotResponsePojo(ClusterizationType.GEO, "handle", snapshot);

        MapF<AlbumType, Integer> indexAlbums = pojo.index.get().items.last().albums;
        ListF<SnapshotClusterItemPojo> clusterItems = pojo.clusters.get().items.last().items;

        if (withAlbums) {
            Assert.some(1, indexAlbums.getO(AlbumType.BEAUTIFUL));
            Assert.some(1, indexAlbums.getO(AlbumType.UNBEAUTIFUL));
            Assert.equals(1, clusterItems.count(item -> item.fields.albums.isSome(
                    Cf.set(AlbumType.BEAUTIFUL, AlbumType.NONPHOTOUNLIM))));
            Assert.equals(1, clusterItems.count(item -> item.fields.albums.isSome(
                    Cf.set(AlbumType.UNBEAUTIFUL, AlbumType.NONPHOTOUNLIM))));

            ListF<SnapshotAlbumPojo> albumItems = pojo.albums.get().items;

            Assert.hasSize(3, albumItems);
            Assert.exists(albumItems, albumPojo -> albumPojo.album == AlbumType.BEAUTIFUL
                    && albumPojo.count == 2
                    && albumPojo.getPreviews().equals(Cf.list("/disk/new beauty.JPG", "/disk/old beauty.JPG")));
            Assert.exists(albumItems, albumPojo -> albumPojo.album == AlbumType.UNBEAUTIFUL && albumPojo.count == 1);
            Assert.exists(albumItems, albumPojo -> albumPojo.album == AlbumType.NONPHOTOUNLIM && albumPojo.count == 3);

        } else {
            Assert.isEmpty(indexAlbums);
            Assert.forAll(clusterItems, item -> !item.fields.albums.isPresent());
            Assert.isEmpty(pojo.albums);
        }
    }

    private IndexedCluster getIndexedCluster(PhotoViewLuceneClusterPojo cluster) {
        return IndexedCluster.consFromLuceneData(
                ClusterId.Format.ADVANCED, cluster.min, cluster.max, cluster.size, Option.empty(), cluster.mergedDocs
        );
    }

    private PhotoViewLuceneInfoPojo createRandomPhotoWithBeauty(String name, double beauty) {
        return new PhotoViewLuceneInfoPojo(Random2.R.nextAlnum(10),
                "/disk/" + name + ".JPG", new Instant(1423046235000L),
                Option.empty(), Option.empty(), Option.of(640), Option.of(480), Option.of(beauty), 1L,
                Option.of("9"), Option.empty(), Option.empty(), false);
    }

    private Snapshot getSnapshot(Database db, Option<SetF<String>> collectionIdsO) {
        return dataApiStorageManager.getSnapshotO(
                testUid, db.handleValue(), db.rev, collectionIdsO, SqlLimits.all()).get().getSnapshot();
    }

    @Test
    public void albumDeltaTest() {
        Assert.equals(AlbumType.BEAUTIFUL, AlbumsToDeltaMapper.parseAlbumTypeFromField("albumBeautiful"));
    }
}
