package ru.yandex.chemodan.app.smartcache.client.actions;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotWithSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.dataapi.web.convert.ConvertToDataapiUser;
import ru.yandex.chemodan.app.smartcache.NS;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.ResponsePojoMappers;
import ru.yandex.chemodan.app.smartcache.client.actions.pojo.SnapshotResponsePojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizationType;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.utils.SqlLimitsHelper;
import ru.yandex.chemodan.log.utils.ExtraRequestLogFieldsUtil;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.support.PublicCloneableActionSupport;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author osidorkin
 */
@Action(
        value = @Action.Alias(value = "get-smartcache-snapshot", namespace = NS.SMARTCACHE),
        description = "Получить снапшот базы")
@Path(value = "/smartcache-snapshot", methods = HttpMethod.GET)
public class GetSmartCacheSnapshotAction extends PublicCloneableActionSupport {

    @RequestParam(value = "__uid", customConverter = ConvertToDataapiUser.class)
    private DataApiUserId user;
    @RequestParam
    private String photosliceId;
    @RequestParam
    private Option<Long> rev;
    @RequestListParam(value = "clusterIds", required = false)
    private ListF<String> clusterIds;
    @RequestListParam(value = "fields", required = false)
    private ListF<String> fields;
    @RequestParam
    private Option<Integer> offset;
    @RequestParam
    private Option<Integer> limit;

    @SpecialParam
    private HttpServletRequestX request;

    private final DataApiStorageManager dataApiStorageManager;
    private final CleanupManager cleanupManager;


    public GetSmartCacheSnapshotAction(DataApiStorageManager dataApiStorageManager, CleanupManager cleanupManager) {
        this.dataApiStorageManager = dataApiStorageManager;
        this.cleanupManager = cleanupManager;
    }

    @Override
    public SnapshotResponsePojo execute() {

        boolean loadClusters = true;
        boolean loadIndex = true;
        boolean loadAlbums = true;

        if (fields.isNotEmpty()) {
            if (!fields.exists(s -> s.startsWith("index"))) {
                loadIndex = false;
            }
            if (!fields.exists(s -> s.startsWith("clusters"))) {
                loadClusters = false;
            }
            if (!fields.exists(s -> s.startsWith("albums"))) {
                loadAlbums = false;
            }
        }

        SnapshotWithSource snapshot = loadSnapshotO(loadIndex, loadClusters, loadAlbums)
                .getOrThrow(() -> new NotFoundException("Snapshot not found"));

        Database database = snapshot.getSnapshot().database;
        cleanupManager.revisionRetrieved(user, photosliceId, database.rev);
        cleanupManager.deltaRetrieved(user, database.handleValue(), database.rev, database.rev);

        ExtraRequestLogFieldsUtil.addField(request, "snapshot_source", snapshot.getSource());
        return ResponsePojoMappers.snapshotResponsePojo(ClusterizationType.GEO, photosliceId, snapshot.getSnapshot());
    }

    private Option<SnapshotWithSource> loadSnapshotO(boolean loadIndex, boolean loadClusters, boolean loadAlbums) {

        SetF<String> nonClusterCollections = Option.when(loadIndex, IndexToDeltaMapper.INDEX_COLLECTION_ID)
                .plus(Option.when(loadAlbums, AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID)).unique();

        if (!loadClusters) {
            SqlLimits limits = nonClusterCollections.isEmpty()
                    ? SqlLimits.all()
                    : SqlLimitsHelper.getLimits(limit, offset);
            return dataApiStorageManager.getSnapshotO(user, photosliceId, rev,
                    Option.of(nonClusterCollections), limits);
        }
        Option<SetF<String>> clusterIdsO = Option.when(clusterIds.isNotEmpty(),
                () -> CollectionIdsResolver.resolve(clusterIds).plus(nonClusterCollections));

        return dataApiStorageManager.getSnapshotO(user, photosliceId, rev, clusterIdsO, SqlLimits.all());
    }
}
