package ru.yandex.chemodan.app.smartcache.client.actions.pojo;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojo;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojoRow;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange.FieldChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.smartcache.AlbumsUtils;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizationType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumFields;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.ClusterKeys;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoFields;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoRecordId;
import ru.yandex.chemodan.app.smartcache.worker.utils.SortedSplitter;
import ru.yandex.inside.utils.Language;

/**
 * @author osidorkin
 */
public class ResponsePojoMappers {

    public static SnapshotResponsePojo snapshotResponsePojo(ClusterizationType clusterizationType, String handle, Snapshot snapshot) {
        SnapshotPojo pojo = snapshot.toPojo();
        boolean withAlbums = AlbumsUtils.isWithAlbums(snapshot);
        ListF<SnapshotPojoRow> rows = pojo.objects;
        Tuple2List<String, ListF<SnapshotPojoRow>> clusters = SortedSplitter.splitCollections(rows, row -> row.collectionId);
        Option<ListF<SnapshotIndexItemPojo>> indexItemsO = Option.empty();
        Option<ListF<SnapshotClusterPojo>> clustersO = Option.empty();
        Option<ListF<SnapshotAlbumPojo>> albumsMetaO = Option.empty();
        if (clusters.isNotEmpty()) {
            Tuple2<String, ListF<SnapshotPojoRow>> lastCluster = clusters.get(0);
            if (IndexToDeltaMapper.INDEX_COLLECTION_ID.equals(lastCluster.get1())) {
                indexItemsO = Option.of(lastCluster.get2().map(row -> snapshotIndexItemPojo(row, withAlbums)));
                clusters = clusters.subList(1, clusters.size());
            }
            if (clusters.isNotEmpty()) {
                lastCluster = clusters.get(0);
                if (AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID.equals(lastCluster.get1())) {
                    if (withAlbums) {
                        albumsMetaO = Option.of(lastCluster.get2().map(ResponsePojoMappers::snapshotAlbumMetaPojo));
                    }
                    clusters = clusters.subList(1, clusters.size());
                }
            }
            if (clusters.isNotEmpty()) {
                clustersO = Option.of(clusters.map((id, clusterRows) ->
                        snapshotClusterPojo(id, clusterRows, withAlbums)));
            }
        }
        return new SnapshotResponsePojo(clusterizationType, handle, pojo.rev, indexItemsO, clustersO, albumsMetaO);
    }

    private static SnapshotIndexItemPojo snapshotIndexItemPojo(SnapshotPojoRow indexRow, boolean withAlbums) {
        return SnapshotIndexItemPojo.cons(
                IndexToDeltaMapper.retrieveDisplayedCluster(indexRow.recordId, indexRow.getData(), withAlbums)
        );
    }

    private static SnapshotAlbumPojo snapshotAlbumMetaPojo(SnapshotPojoRow albumRow) {
        AlbumType album = AlbumsToDeltaMapper.parseAlbumTypeFromRecordId(albumRow.recordId);
        ListF<String> previews = albumRow.getData().getO(AlbumFields.PREVIEWS.name).map(DataField::listValue)
                .getOrElse(Cf.list()).map(DataField::stringValue);
        int count = albumRow.getData().getO(AlbumFields.COUNT.name)
                .map(DataField::integerValue).map(Long::intValue).getOrElse(0);
        return new SnapshotAlbumPojo(album, previews, count);
    }

    private static SnapshotClusterPojo snapshotClusterPojo(
            String clusterId, ListF<SnapshotPojoRow> clusterRows, boolean withAlbums)
    {
        return new SnapshotClusterPojo(
                ClusterId.formatForView(clusterId), clusterRows.map(row -> snapshotClusterItemPojo(row, withAlbums)));
    }

    private static SnapshotClusterItemPojo snapshotClusterItemPojo(SnapshotPojoRow row, boolean withAlbums) {
        try {
            return new SnapshotClusterItemPojo(
                    PhotoRecordId.formatForView(row.recordId),
                    photoFieldsPojo(row.getData(), withAlbums).get());

        } catch (RuntimeException e) {
            throw new RuntimeException(row.collectionId + " " + row.recordId, e);
        }
    }

    public static DeltaListResponsePojo deltaListResponsePojo(ListF<Delta> deltas, long dbRev, boolean withAlbums) {
        long lastRev = deltas.last().newRev.get();
        long total = dbRev - deltas.first().rev.get();
        return new DeltaListResponsePojo(lastRev, total, deltas.size(),
                deltas.map(delta -> deltaPojo(delta, withAlbums)));
    }

    private static <T> Option<ListF<T>> notEmptyListO(ListF<T> list) {
        return Option.when(list.isNotEmpty(), list);
    }

    private static DeltaPojo deltaPojo(Delta delta, boolean withAlbums) {
        ListF<RecordChange> recordChanges = delta.changes;
        ListF<DeltaIndexChangePojo> indexChanges = Cf.arrayList();
        ListF<DeltaClusterChangePojo> clusterChanges = Cf.arrayList();
        ListF<DeltaAlbumsChangePojo> albumsChanges = Cf.arrayList();
        for (RecordChange recordChange: recordChanges) {
            String collectionId = recordChange.collectionId;
            if (IndexToDeltaMapper.INDEX_COLLECTION_ID.equals(collectionId)) {
                indexChanges.add(deltaIndexChangePojo(recordChange));
                continue;
            } else if (AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID.equals(collectionId)) {
                albumsChanges.add(deltaAlbumsChangePojo(recordChange));
                continue;
            }
            clusterChanges.add(deltaClusterChangePojo(recordChange, withAlbums));
        }

        return new DeltaPojo(delta.rev.get(), delta.newRev.get(),
                notEmptyListO(indexChanges), notEmptyListO(clusterChanges), notEmptyListO(albumsChanges));
    }

    public static DeltaListResponsePojo deltaListResponseEmptyPojo(long rev) {
        return new DeltaListResponsePojo(rev, 0, 0, Cf.list());
    }

    private static DeltaIndexChangePojo deltaIndexChangePojo(RecordChange recordChange) {
        RecordChangeType type = recordChange.type;
        String clusterId = ClusterId.formatForView(recordChange.recordId);

        if (type == RecordChangeType.DELETE) {
            return new DeltaIndexChangePojo(type, clusterId, Option.empty());
        }
        return new DeltaIndexChangePojo(type, clusterId, Option.of(deltaIndexFieldChangePojo(recordChange.fieldChanges)));
    }

    private static DeltaIndexFieldChangePojo deltaIndexFieldChangePojo(ListF<FieldChange> changes) {
        Option<Long> itemsCountO = Option.empty();
        Option<Instant> fromO = Option.empty();
        Option<Instant> toO = Option.empty();
        Option<DeltaLocalityChangePojo> cityO = Option.empty();
        ListF<DeltaPlacesListChangePojo> places = Cf.arrayList();
        ListF<DeltaAlbumCounterChangePojo> albums = Cf.arrayList();
        for (FieldChange fieldChange: changes) {
            switch (fieldChange.key) {
            case ClusterKeys.PHOTOS_COUNT:
                itemsCountO = Option.of(fieldChange.getValue().integerValue());
                break;
            case ClusterKeys.FROM_INSTANT:
                fromO = Option.of(fieldChange.getValue().timestampValue());
                break;
            case ClusterKeys.TO_INSTANT:
                toO = Option.of(fieldChange.getValue().timestampValue());
                break;
            case ClusterKeys.LOCALITY:
                cityO = Option.of(DeltaLocalityChangePojo.consFromChange(fieldChange));
                break;
            case ClusterKeys.PLACES:
                places.addAll(placesChanges(fieldChange));
                break;
            default:
                if (fieldChange.key.startsWith(ClusterKeys.ALBUM_PREFIX)) {
                    albums.add(DeltaAlbumCounterChangePojo.consFromChange(fieldChange));
                }
            }
        }
        return new DeltaIndexFieldChangePojo(itemsCountO, fromO, toO, cityO,
                notEmptyListO(places), notEmptyListO(albums));
    }

    private static DeltaAlbumsChangePojo deltaAlbumsChangePojo(RecordChange recordChange) {
        RecordChangeType type = recordChange.type;
        AlbumType album = AlbumsToDeltaMapper.parseAlbumTypeFromRecordId(recordChange.recordId);
        if (type == RecordChangeType.DELETE) {
            return new DeltaAlbumsChangePojo(type, album, Option.empty());
        }
        return new DeltaAlbumsChangePojo(type, album, Option.of(deltaAlbumsFieldChangePojo(recordChange.fieldChanges)));
    }

    private static DeltaAlbumsFieldChangePojo deltaAlbumsFieldChangePojo(ListF<FieldChange> changes) {
        Option<ListF<String>> previewsO = Option.empty();
        Option<Integer> countO = Option.empty();
        for (FieldChange fieldChange: changes) {
            if (AlbumFields.PREVIEWS.name.equals(fieldChange.key)) {
                previewsO = Option.of(fieldChange.getValue().listValue().map(DataField::stringValue));
            } else if (AlbumFields.COUNT.name.equals(fieldChange.key)) {
                countO = Option.of(fieldChange.getValue().integerValue().intValue());
            }
        }
        return new DeltaAlbumsFieldChangePojo(previewsO, countO);
    }

    private static ListF<DeltaPlacesListChangePojo> placesChanges(FieldChange fieldChange) {
        if (fieldChange.type == FieldChangeType.PUT) {
            ListF<DataField> values = fieldChange.getValue().listValue();
            ListF<DeltaPlacesListChangePojo> res = Cf.arrayList(values.size());
            for (int i = 0; i < values.size(); i++) {
                MapF<Language, String> resources = IndexToDeltaMapper
                        .fieldToLocalizedStringDictionary(values.get(i)).toMap();
                res.add(new DeltaPlacesListChangePojo(FieldChangeType.INSERT_LIST_ITEM, i, Option.of(resources)));
            }
            return res;
        }
        if (fieldChange.type == FieldChangeType.DELETE_LIST_ITEM) {
            return Cf.list(new DeltaPlacesListChangePojo(fieldChange.type, fieldChange.listIndex.get(), Option.empty()));
        }
        return Cf.list(placesChange(fieldChange, fieldChange.getValue()));
    }

    private static DeltaPlacesListChangePojo placesChange(FieldChange fieldChange, DataField value) {
        MapF<Language, String> resources = IndexToDeltaMapper.fieldToLocalizedStringDictionary(value).toMap();
        return new DeltaPlacesListChangePojo(fieldChange.type, fieldChange.listIndex.get(), Option.of(resources));
    }

    private static DeltaClusterChangePojo deltaClusterChangePojo(RecordChange recordChange, boolean withAlbums) {
        MapF<String, DataField> changes = recordChange.fieldChanges
                .filter(ch -> ch.type == FieldChangeType.PUT)
                .toMap(ch -> ch.key, FieldChange::getValue);

        return new DeltaClusterChangePojo(recordChange.type,
                ClusterId.formatForView(recordChange.collectionId),
                PhotoRecordId.formatForView(recordChange.recordId),
                photoFieldsPojo(changes, withAlbums));
    }

    private static Option<PhotoFieldsPojo> photoFieldsPojo(MapF<String, DataField> source, boolean withAlbums) {
        if (!source.containsKeyTs(PhotoFields.PATH.name)) {
            return Option.empty();
        }
        return Option.of(new PhotoFieldsPojo(
                PhotoFields.PATH.get(source),
                PhotoFields.WIDTH.getO(source),
                PhotoFields.HEIGHT.getO(source),
                PhotoFields.BEAUTY.getO(source),
                Option.when(withAlbums, PhotoFields.ALBUMS.getO(source).getOrElse(Cf.set()))));
    }
}
