package ru.yandex.chemodan.app.smartcache.worker;

import java.net.URI;
import java.util.concurrent.ExecutorService;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.migration.DsMigrationDataSourceRegistry;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManagerImpl;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizerClientContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizerManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManagerContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.processing.GeoPlaceLocalizedNames;
import ru.yandex.chemodan.app.smartcache.worker.processing.GeocoderManager;
import ru.yandex.chemodan.app.smartcache.worker.processing.PhotosliceProcessingManager;
import ru.yandex.chemodan.app.smartcache.worker.processing.PhotosliceTransitionManager;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.geosearch.GeosearchClient;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;

import static ru.yandex.chemodan.app.dataapi.core.DatabasesContextConfiguration.DATABASE_MANAGER_EXECUTOR_SERVICE;

/**
 * @author osidorkin
 */
@Configuration
@Import({
    DataApiStorageManagerContextConfiguration.class,
    ClusterizerClientContextConfiguration.class,
    SmartCacheDynamicPropertiesContextConfiguration.class,
})
public class SmartCacheContextConfiguration {

    public static final String SMARTCACHE_WORKER_USER_AGENT = "disk-smartcache-worker";

    @Bean
    public PhotosliceProcessingManager photosliceProcessingManager(DataApiStorageManager dataApiStorageManager,
            ClusterizerManager clusterizerManager, GeocoderManager geocoderManager,
            CleanupManager cleanupManager, PgBazingaStorage bazingaStorage,
            MpfsClient mpfsClient, PhotosliceTransitionManager photosliceTransitionManager)
    {
        return new PhotosliceProcessingManager(dataApiStorageManager, clusterizerManager,
                geocoderManager, cleanupManager, bazingaStorage, mpfsClient, photosliceTransitionManager);
    }

    @Bean
    public GeocoderManager geocoderManager(
            @Value("${cache.geocoder.ttl}") Duration cacheTtl,
            @Value("${cache.geocoder.size}") int cacheSize,
            @Value("${cache.geocoder.concurrency}") int concurrency,
            GeosearchClient geosearchClient,
            DynamicPropertyManager dynamicPropertyManager)
    {
        MeteredCache<Coordinates, GeoPlaceLocalizedNames> cache =
                new MeteredCache<>(cacheSize, concurrency, cacheTtl);

        MonicaStaticRegistry.register(cache, "geocoder");

        return new GeocoderManager(geosearchClient, dynamicPropertyManager, cache);
    }

    @Bean
    public GeosearchClient geosearchClient(
            @Value("${geosearch.uri}") URI geosearchUri,
            @Value("${geosearch.origin}") String origin)
    {
        return new GeosearchClient(geosearchUri, origin, geosearchHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("geosearch")
    public HttpClientConfigurator geosearchHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public PhotosliceTransitionManager photosliceTransitionManager(
            BazingaTaskManager bazingaTaskManager,
            ClusterizerManager clusterizerManager,
            DataRecordsJdbcDao dataRecordsJdbcDao,
            DataApiStorageManager dataApiStorageManager,
            UserMetaManager userMetaManager,

            DsMigrationDataSourceRegistry dataSourceRegistry,
            DatabaseLimiter databaseLimiter,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            CompositeApplicationManager appManager,
            AppSettingsRegistry appSettingsRegistry,
            MdsSnapshotReferenceManager mdsSnapshotReferenceManager,
            @Qualifier(DATABASE_MANAGER_EXECUTOR_SERVICE)
            ExecutorService databaseManagerExecutorService)
    {
        DataApiManager noXivaDataApiManager = new DataApiManagerImpl(
                dataSourceRegistry,
                databaseLimiter,
                externalDatabasesRegistry,
                appManager,
                appSettingsRegistry,
                mdsSnapshotReferenceManager,
                Cf.list(), Cf.list(),
                databaseManagerExecutorService
        );
        return new PhotosliceTransitionManager(bazingaTaskManager,
                clusterizerManager, noXivaDataApiManager, dataRecordsJdbcDao,
                dataApiStorageManager, userMetaManager);
    }
}
