package ru.yandex.chemodan.app.smartcache.worker;

import java.util.concurrent.ThreadFactory;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.dataapi.maintenance.DeleteOldSnapshotReferencesCronTask;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.DeltaAccessTimeMdao;
import ru.yandex.chemodan.app.smartcache.worker.processing.PhotosliceProcessingManager;
import ru.yandex.chemodan.app.smartcache.worker.processing.PhotosliceTransitionManager;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.DeleteDeltaTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.InitAlbumsTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.RemoveExpiredSnapshotTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.SmcDeleteOldSnapshotReferenceByShardTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdateClusterPlacesTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceBlockerTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceSubmitterTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceTask;
import ru.yandex.chemodan.app.smartcache.worker.utils.SmartCacheCronTaskWrapper;
import ru.yandex.commune.bazinga.BazingaConfiguration;
import ru.yandex.commune.bazinga.BazingaConfigurator;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.BazingaUtils;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.tasks.BazingaHeartbeatTask;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * @author dbrylev
 */
@Configuration
public class SmartCacheTasksContextConfiguration {

    private final DynamicProperty<Boolean> parallelDeleteOldSnapshotReferenceEnabled =
            new DynamicProperty<>("smartcache-delete-old-snapshot-reference-parallel-enabled", false);

    private final DynamicProperty<Integer> parallelDeleteOldSnapshotReferenceThreadCount =
            new DynamicProperty<>("smartcache-delete-old-snapshot-reference-parallel-thread-count", 5);

    @Bean
    public CronTask deleteOldSnapshotReferencesCronTask(
            @Value("${dataapi.snapshot.deletion-task.execution-period}")
            Duration executionPeriod,
            @Value("${dataapi.snapshot.deletion-task.timeout}")
            Duration timeout,
            MdsSnapshotReferenceManager mdsSnapshotReferenceManager, @Qualifier("dataShardManager") ShardManager2 dataShardManager,
            BazingaTaskManager bazingaTaskManager)
    {
        return new SmartCacheCronTaskWrapper(
                new DeleteOldSnapshotReferencesCronTask(executionPeriod, timeout, mdsSnapshotReferenceManager,
                        parallelDeleteOldSnapshotReferenceEnabled::get, dataShardManager, bazingaTaskManager,
                        SmcDeleteOldSnapshotReferenceByShardTask::new))
        {
            public TaskId id() {
                return new TaskId(BazingaUtils.simpleClassNameToTaskId(DeleteOldSnapshotReferencesCronTask.class));
            }
        };
    }

    @Bean
    public CronTask removeExpiredSnapshotTask(CleanupManager cleanupManager, BazingaTaskManager bazingaTaskManager,
                                              BazingaStorage bazingaStorage, DeltaAccessTimeMdao deltaAccessTimeMdao) {
        return new SmartCacheCronTaskWrapper(new RemoveExpiredSnapshotTask(cleanupManager, bazingaTaskManager,
                bazingaStorage, deltaAccessTimeMdao));
    }

    @Bean
    public CronTask bazingaHeartbeatTask(BazingaConfiguration bazingaConfiguration, ZkManager zkManager) {
        return new SmartCacheCronTaskWrapper(new BazingaHeartbeatTask(
                bazingaConfiguration,
                BazingaConfigurator.zkBazingaConfigurationStorage(bazingaConfiguration, zkManager)))
        {
            public TaskId id() {
                return new TaskId("bazingaHeartbeat.smartcache");
            }
        };
    }

    @Bean
    public UpdateClusterPlacesTask updateClusterPlacesTask(PhotosliceProcessingManager photosliceProcessingManager) {
        return new UpdateClusterPlacesTask(photosliceProcessingManager);
    }

    @Bean
    public UpdatePhotosliceTask updatePhotosliceTask(PhotosliceProcessingManager photosliceProcessingManager) {
        return new UpdatePhotosliceTask(photosliceProcessingManager);
    }

    @Bean
    public UpdatePhotosliceSubmitterTask updatePhotosliceSubmitterTask(PgBazingaStorage bazingaStorage) {
        return new UpdatePhotosliceSubmitterTask(bazingaStorage);
    }

    @Bean
    public UpdatePhotosliceBlockerTask updatePhotosliceBlockerTask() {
        return new UpdatePhotosliceBlockerTask();
    }

    @Bean
    public InitAlbumsTask initAlbumsTask(PhotosliceTransitionManager photosliceTransitionManager,
            PhotosliceProcessingManager photosliceProcessingManager)
    {
        return new InitAlbumsTask(photosliceTransitionManager, photosliceProcessingManager);
    }

    @Bean
    public ThreadFactory deleteSnapshotsThreadFactory() {
        return new ThreadNameIndexThreadFactory("smartcache-delete-old-snapshot-reference");
    }

    @Bean
    public SmcDeleteOldSnapshotReferenceByShardTask smcDeleteOldSnapshotReferenceByShardTask(@Qualifier("dataShardManager") ShardManager2 dataShardManager,
                                                                                       MdsSnapshotReferenceManager mdsSnapshotReferenceManager,
                                                                                       @Qualifier("deleteSnapshotsThreadFactory") ThreadFactory deleteSnapshotsThreadFactory)
    {
        return new SmcDeleteOldSnapshotReferenceByShardTask(mdsSnapshotReferenceManager, dataShardManager,
                parallelDeleteOldSnapshotReferenceThreadCount::get, deleteSnapshotsThreadFactory);
    }

    @Bean
    public DeleteDeltaTask deleteDeltaTask(CleanupManager cleanupManager) {
        return new DeleteDeltaTask(cleanupManager);
    }
}
