package ru.yandex.chemodan.app.smartcache.worker.clusterizer;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
public class ClusterizerDuplicatesFilterTest {

    @Test
    public void commonCase() {
        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(getFirstClusterPhotos(), getSecondClusterPhotos())
                .map(ClusterizerDuplicatesFilterTest::createCluster);
        Assert.assertListsEqual(clusters, ClusterizerDuplicatesFilter.removeClusterPathDuplicates(clusters));
    }

    @Test
    public void duplicatesInsideClusterCase() {
        ListF<PhotoViewLuceneInfoPojo> firstCluster = Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 1L))
                .plus(getFirstClusterPhotos()).plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L)));

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(firstCluster, getSecondClusterPhotos())
                .map(ClusterizerDuplicatesFilterTest::createCluster);

        ListF<PhotoViewLuceneClusterPojo> etalon = Cf.list(getFirstClusterPhotos()
                    .plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L))),
                getSecondClusterPhotos())
                .map(ClusterizerDuplicatesFilterTest::createCluster);

        Assert.assertListsEqual(etalon, ClusterizerDuplicatesFilter.removeClusterPathDuplicates(clusters));
    }

    @Test
    public void duplicatesBetweenClustersCase() {
        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 1L)).plus(getFirstClusterPhotos()),
                getSecondClusterPhotos().plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L))))
                .map(ClusterizerDuplicatesFilterTest::createCluster);

        ListF<PhotoViewLuceneClusterPojo> etalon = Cf.list(getFirstClusterPhotos(),
                getSecondClusterPhotos().plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L))))
                .map(ClusterizerDuplicatesFilterTest::createCluster);

        Assert.assertListsEqual(etalon, ClusterizerDuplicatesFilter.removeClusterPathDuplicates(clusters));
    }

    @Test
    public void duplicatesEmptyClustersCase() {
        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 1L)),
                getSecondClusterPhotos().plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L))))
                .map(ClusterizerDuplicatesFilterTest::createCluster);

        ListF<PhotoViewLuceneClusterPojo> etalon = Cf.list(createCluster(
                getSecondClusterPhotos().plus(Cf.list(getDuplicatePhoto("/disk/Фотокамера/1", 2L)))));

        Assert.assertListsEqual(etalon, ClusterizerDuplicatesFilter.removeClusterPathDuplicates(clusters));
    }

    @Test
    public void sameVersionDuplicate() {
        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(createCluster(Cf.<PhotoViewLuceneInfoPojo>list()
                .plus1(getDuplicatePhoto("/disk/Фотокамера/1", 1L))
                .plus(getFirstClusterPhotos())
                .plus1(getDuplicatePhoto("/disk/Фотокамера/1", 1L))));

        ListF<PhotoViewLuceneClusterPojo> expected = Cf.list(createCluster(Cf.<PhotoViewLuceneInfoPojo>list()
                .plus1(getDuplicatePhoto("/disk/Фотокамера/1", 1L))
                .plus(getFirstClusterPhotos())));

        Assert.assertListsEqual(expected, ClusterizerDuplicatesFilter.removeClusterPathDuplicates(clusters));
    }


    private static PhotoViewLuceneClusterPojo createCluster(ListF<PhotoViewLuceneInfoPojo> photos) {
        return new PhotoViewLuceneClusterPojo(photos.size(), photos.last().date, photos.first().date, photos);
    }

    private static ListF<PhotoViewLuceneInfoPojo> getFirstClusterPhotos() {
        return Cf.list(
                new PhotoViewLuceneInfoPojo("f236590b7ceaab57541d4e707d25ca91ab6135c150e3bf7c54fd161a45487602",
                        "/disk/Фотокамера/2012-06-06 23-20-58.JPG",
                        instant(1338996058000L), Option.empty(), Option.empty(), 1L),
                new PhotoViewLuceneInfoPojo("b4a173f8e5a084df8294ccb68b3569f5fece40c419ed35df5f1e2ece723b15b0",
                        "/disk/Фотокамера/2012-06-06 23-20-54.JPG",
                        instant(1338996054000L), Option.empty(), Option.empty(), 2L)
               );
    }

    private static PhotoViewLuceneInfoPojo getDuplicatePhoto(String key, long version) {
        return new PhotoViewLuceneInfoPojo("b25eef7508e634d14ebae1570481adb5968ba070c72e19ad4288f36c1c593e59",
                key, instant(1326833018000L), Option.empty(), Option.empty(), version);
    }

    private static ListF<PhotoViewLuceneInfoPojo> getSecondClusterPhotos() {
        return Cf.list(
                new PhotoViewLuceneInfoPojo("020929811ec3edc2a44e4c90338fadf246713e71b448a9c7b4281332cf393204",
                        "/disk/Святое 29-30.11.2014/Разуваем пламя газовой горелкой. Андрей слева.JPG", instant(1417807199000L),
                        Option.of(42.310467), Option.of(55.640378), 1L),
                new PhotoViewLuceneInfoPojo("d45b2a49f9a628827939504b10c6a3b7501c6e9a4ae11a5bb964c4708e2d9c97",
                        "/disk/Святое 29-30.11.2014/Гамиль.JPG", instant(1417807199000L),
                        Option.of(42.310478), Option.of(55.640719), 2L),
                new PhotoViewLuceneInfoPojo("44ab0a7b494838465c7511a806e79ca7024f3b9dda94da1df49266649b7ab27b",
                        "/disk/Фотокамера/2015-01-20 21-05-30.MP4", instant(1421777202000L),
                        Option.empty(), Option.empty(), 3L),
                new PhotoViewLuceneInfoPojo("113b9c6f96157a39b96280b10959ecdacff7be025957d7f50f0af146b2d27690",
                        "/disk/Социальные сети/Vkontakte/Фото со мной/2753165.jpg", instant(1176228688000L),
                        Option.empty(), Option.empty(), 4L)
                );
    }

    private static Instant instant(long instant) {
        return new Instant(instant);
    }
}
