package ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.smartcache.worker.processing.CoordinatesAware;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultToString;

/**
 * @author osidorkin
 */
@AllArgsConstructor
@BenderBindAllFields
public class PhotoViewLuceneInfoPojo extends DefaultToString implements CoordinatesAware {
    public final String id;
    public final String key;
    public final Instant date;
    public final Option<Double> longitude;
    public final Option<Double> latitude;
    public final Option<Integer> width;
    public final Option<Integer> height;
    @BenderPart(name = "cost_disk_aethetic_0", strictName = true)
    public final Option<Double> beauty;
    public final long version;

    public final Option<String> mediatype;
    @BenderPart(name = "photoslice_album_type", strictName = true)
    public final Option<AlbumType> cameraOrScreenshot;
    @BenderPart(name = "albums_exclusions", strictName = true)
    public final Option<String> exclusion;

    @BenderIgnore
    private transient boolean isWithNoAlbums;
    @BenderIgnore
    private transient SetF<AlbumType> albums;

    private static ListF<AlbumType> parseExclusion(Option<String> exclusion) {
        return exclusion.flatMap(value -> Cf.x(value.split("\\n"))).filterMap(AlbumType::valueOfSafe);
    }

    public PhotoViewLuceneInfoPojo(String id, String key, Instant date,
            Option<Double> longitude, Option<Double> latitude, Option<Integer> width,
            Option<Integer> height, Option<Double> beauty, long version,
            Option<String> mediatype, Option<AlbumType> cameraOrScreenshot, Option<String> exclusion,
            boolean isWithNoAlbums)
    {
        this(id, key, date, longitude, latitude, width, height, beauty, version,
                mediatype, cameraOrScreenshot, exclusion, isWithNoAlbums,
                isWithNoAlbums ? Cf.set() : AlbumType.getMatching(
                        key, beauty, mediatype, cameraOrScreenshot, parseExclusion(exclusion))
        );
    }

    public PhotoViewLuceneInfoPojo(
            String id, String key, Instant date, Option<Double> longitude, Option<Double> latitude, long version)
    {
        this(id, key, date, longitude, latitude, Option.empty(), Option.empty(), Option.empty(),
                version, Option.empty(), Option.empty(), Option.empty(), false);
    }

    public PhotoViewLuceneInfoPojo withNoAlbums() {
        return new PhotoViewLuceneInfoPojo(id, key, date, longitude, latitude, width, height, beauty,
                version, mediatype, cameraOrScreenshot, exclusion, true);
    }

    @Override
    public Option<Double> getLongitudeO() {
        return longitude;
    }

    @Override
    public Option<Double> getLatitudeO() {
        return latitude;
    }

    public boolean isWithAlbums() {
        return !isWithNoAlbums;
    }

    public SetF<AlbumType> getAlbums() {
        if (albums == null) {
            albums = isWithNoAlbums
                    ? Cf.set() : AlbumType.getMatching(
                            key, beauty, mediatype, cameraOrScreenshot, parseExclusion(exclusion));
        }
        return albums;
    }

    private StoredPojo toStoredPojo() {
        return new StoredPojo(id, key, date, longitude, latitude, width, height, beauty, version, getAlbums());
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof PhotoViewLuceneInfoPojo)) {
            return false;
        }
        PhotoViewLuceneInfoPojo other = (PhotoViewLuceneInfoPojo) obj;
        return this.toStoredPojo().equals(other.toStoredPojo());
    }

    @Override
    public int hashCode() {
        return toStoredPojo().hashCode();
    }

    @AllArgsConstructor
    @Data
    private static class StoredPojo {
        public final String id;
        public final String key;
        public final Instant date;
        public final Option<Double> longitude;
        public final Option<Double> latitude;
        public final Option<Integer> width;
        public final Option<Integer> height;
        public final Option<Double> beauty;
        public final long version;
        public final SetF<AlbumType> albums;
    }
}
