package ru.yandex.chemodan.app.smartcache.worker.dataapi;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoRecordId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.TimestampFormatter;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
public class ClusterId extends DefaultObject {

    public enum Format {
        STALE(TimestampFormatter.NUMERIC, PhotoRecordId.Format.STALE),
        ADVANCED(TimestampFormatter.PRINTED, PhotoRecordId.Format.ADVANCED);

        private final TimestampFormatter formatter;
        private final PhotoRecordId.Format photoIdFormat;

        Format(TimestampFormatter formatter, PhotoRecordId.Format photoIdFormat) {
            this.formatter = formatter;
            this.photoIdFormat = photoIdFormat;
        }

        public ClusterId cons(Instant from, Instant to) {
            return new ClusterId(from, to, this);
        }

        private ClusterId parse(String value) {
            String[] parts = StringUtils.split(value, '_');

            return cons(new Instant(formatter.parse(parts[0])), new Instant(formatter.parse(parts[1])));
        }

        private String format(Instant from, Instant to) {
            return formatter.format(from) + "_" + formatter.format(to);
        }

        private boolean matches(String value) {
            return formatter.matches(value);
        }

        public static ClusterId.Format forLucene(CollectionF<PhotoViewLuceneClusterPojo> pojos) {
            return pojos.exists(p -> p.min.getMillis() < 0) ? ADVANCED : STALE;
        }

        public static Option<Format> ofSnapshot(CollectionF<IndexedCluster> clusters) {
            return clusters.iterator().nextO().map(c -> c.getId().getFormat());
        }

        public static Option<Format> ofSnapshot(Snapshot snapshot) {
            return snapshot.records().iterator()
                    .find(r -> IndexToDeltaMapper.INDEX_COLLECTION_ID.equals(r.getCollectionId()))
                    .map(r -> ADVANCED.matches(r.getRecordId()) ? ADVANCED : STALE);
        }

        public static ClusterId.Format forUpdate(
                CollectionF<IndexedCluster> clusters, CollectionF<PhotoViewLuceneClusterPojo> pojos)
        {
            return ofSnapshot(clusters).isSome(ADVANCED) || forLucene(pojos) == ADVANCED ? ADVANCED : STALE;
        }

        public boolean isIncompatibleTo(Option<ClusterId.Format> from) {
            return from.isSome(STALE) && this == ADVANCED;
        }
    }

    private final Instant from;
    private final Instant to;
    private final Format format;

    public ClusterId(Instant from, Instant to, Format format) {
        this.from = from;
        this.to = to;
        this.format = format;
    }

    public static ClusterId parse(String value) {
        if (Format.ADVANCED.matches(value)) {
            return Format.ADVANCED.parse(value);
        } else {
            return Format.STALE.parse(value);
        }
    }

    public static String formatForView(String value) {
        if (Format.ADVANCED.matches(value)) {
            ClusterId id = Format.ADVANCED.parse(value);
            return Format.STALE.format(id.from, id.to);

        } else {
            return value;
        }
    }

    public String formatForView() {
        return Format.STALE.format(from, to);
    }

    public String formatForDb() {
        return format.format(from, to);
    }

    public ListF<String> formatAll() {
        return Cf.list(
                Format.STALE.format(from, to),
                Format.ADVANCED.format(from, to));
    }

    public boolean isGreater(Instant from, Instant to) {
        return this.from.isAfter(from) || this.from.isEqual(from) && this.to.isAfter(to);
    }

    public Instant getFrom() {
        return from;
    }

    public Instant getTo() {
        return to;
    }

    public Format getFormat() {
        return format;
    }

    public PhotoRecordId.Format getPhotoIdFormat() {
        return format.photoIdFormat;
    }

    @Override
    public String toString() {
        return formatForDb();
    }
}
