package ru.yandex.chemodan.app.smartcache.worker.dataapi;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId.Format;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class ClusterIdTest {

    @Test
    public void parse() {
        Assert.equals(id(Format.STALE, 1000L, 2000L), ClusterId.parse("0000000000000001000_0000000000000002000"));
        Assert.equals(id(Format.STALE, -1000L, 2000L), ClusterId.parse("-000000000000001000_0000000000000002000"));
        Assert.equals(id(Format.STALE, -2000L, -1000L), ClusterId.parse("-000000000000002000_-000000000000001000"));

        Assert.equals(id(Format.ADVANCED, 1000L, 2000L), ClusterId.parse("19700101000001000Z_19700101000002000Z"));
        Assert.equals(id(Format.ADVANCED, -1000L, 2000L), ClusterId.parse("19691231235959000Z_19700101000002000Z"));
        Assert.equals(id(Format.ADVANCED, -2000L, -1000L), ClusterId.parse("19691231235958000Z_19691231235959000Z"));
    }

    @Test
    public void formatForView() {
        Assert.equals("0000000000000001000_0000000000000002000", id(Format.STALE, 1000L, 2000L).formatForView());
        Assert.equals("-000000000000001000_0000000000000002000", id(Format.STALE, -1000L, 2000L).formatForView());
        Assert.equals("-000000000000002000_-000000000000001000", id(Format.STALE, -2000L, -1000L).formatForView());

        Assert.equals("0000000000000001000_0000000000000002000", id(Format.ADVANCED, 1000L, 2000L).formatForView());
        Assert.equals("-000000000000001000_0000000000000002000", id(Format.ADVANCED, -1000L, 2000L).formatForView());
        Assert.equals("-000000000000002000_-000000000000001000", id(Format.ADVANCED, -2000L, -1000L).formatForView());
    }

    @Test
    public void formatForDb() {
        Assert.equals("0000000000000001000_0000000000000002000", id(Format.STALE, 1000L, 2000L).formatForDb());
        Assert.equals("-000000000000001000_0000000000000002000", id(Format.STALE, -1000L, 2000L).formatForDb());
        Assert.equals("-000000000000002000_-000000000000001000", id(Format.STALE, -2000L, -1000L).formatForDb());

        Assert.equals("19700101000001000Z_19700101000002000Z", id(Format.ADVANCED, 1000L, 2000L).formatForDb());
        Assert.equals("19691231235959000Z_19700101000002000Z", id(Format.ADVANCED, -1000L, 2000L).formatForDb());
        Assert.equals("19691231235958000Z_19691231235959000Z", id(Format.ADVANCED, -2000L, -1000L).formatForDb());
    }

    @Test
    public void reformatForView() {
        Assert.equals("0000000000000001000_0000000000000002000",
                ClusterId.formatForView("0000000000000001000_0000000000000002000"));

        Assert.equals("-000000000000001000_0000000000000002000",
                ClusterId.formatForView("19691231235959000Z_19700101000002000Z"));

        Assert.equals("-000000000000002000_-000000000000001000",
                ClusterId.formatForView("19691231235958000Z_19691231235959000Z"));
    }

    private static ClusterId id(Format format, long from, long to) {
        return new ClusterId(new Instant(from), new Instant(to), format);
    }
}
