package ru.yandex.chemodan.app.smartcache.worker.dataapi;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotWithSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUtils;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.manager.SnapshotSource;
import ru.yandex.chemodan.app.dataapi.web.DeltasGoneException;
import ru.yandex.chemodan.app.smartcache.AlbumsUtils;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author osidorkin
 */
public class DataApiStorageManager {
    public static final AppDatabaseRef PHOTOSLICE_DB_REF = new AppDatabaseRef("smartcache", "photoslice");

    private static final ByIdRecordOrder DEFAULT_SNAPSHOT_ORDER = ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC;

    private final DataApiManager dataApiManager;

    private final DiskDataSource diskDataSource;

    public DataApiStorageManager(DataApiManager dataApiManager, DiskDataSource diskDataSource) {
        this.dataApiManager = dataApiManager;
        this.diskDataSource = diskDataSource;
    }

    public void storeSnapshotIndex(Database indexDatabase, RevisionCheckMode revCheckMode, ListF<RecordChange> snapshot) {
        final long previousRev = indexDatabase.rev;
        Delta delta = new Delta(snapshot);
        dataApiManager.applyDelta(indexDatabase, revCheckMode, delta);
        //Remove the first delta since the snapshot itself is the first delta
        diskDataSource.removeDelta(previousRev, indexDatabase);
    }

    public void storeDelta(DataApiUserId uid, long rev, RevisionCheckMode revCheckMode, ListF<RecordChange> diffs) {
        ListF<Delta> deltas = getBatchedDeltas(diffs, rev);
        dataApiManager.applyDeltas(consDatabaseSpec(uid), rev, revCheckMode, deltas);
    }

    ListF<Delta> getBatchedDeltas(ListF<RecordChange> diffs, long rev) {
        Integer batchSize = DynamicVars.updateSnapshotDiffBatchSize.get();
        final AtomicInteger counter = new AtomicInteger();
        ListF<Delta> deltas;
        return Cf.toList(
                        diffs.stream()
                                .collect(Collectors.groupingBy(it -> counter.getAndIncrement() / batchSize))
                                .values()
                )
                .map(diffList -> new Delta(Cf.toList(diffList)).withRev(rev));
    }

    public void removeDatabase(DataApiUserId uid) {
        dataApiManager.deleteDatabaseIfExists(consDatabaseSpec(uid), DatabaseDeletionMode.REMOVE_COMPLETELY);
    }

    public void removeDatabaseByHandle(DataApiUserId uid, String handle) {
        dataApiManager.deleteDatabaseIfExists(consDbSpecWithHandle(uid, handle),
                DatabaseDeletionMode.REMOVE_COMPLETELY);
    }

    public void removeDatabaseByHandle(DataApiUserId uid, String handle, ChunkRateLimiter rateLimiter) {
        dataApiManager.deleteDatabaseIfExists(consDbSpecWithHandle(uid, handle),
                DatabaseDeletionMode.REMOVE_COMPLETELY, rateLimiter);
    }

    public Snapshot getSnapshot(DataApiUserId uid, CollectionIdCondition collections) {
        return getSnapshotO(uid, collections).getOrThrow(() -> consDatabaseSpec(uid).consNotFound());
    }

    public Option<Snapshot> getSnapshotO(DataApiUserId uid, CollectionIdCondition collections) {
        return getSnapshotO(uid, Option.empty(), collections);
    }

    public Option<Snapshot> getSnapshotO(DataApiUserId uid, Option<String> handle, CollectionIdCondition collections) {
        return dataApiManager.getSnapshotO(
                handle.isPresent() ? consDbSpecWithHandle(uid, handle.get()) : consDatabaseSpec(uid),
                RecordsFilter.DEFAULT
                        .withCollectionIdCond(collections)
                        .withRecordOrder(DEFAULT_SNAPSHOT_ORDER),
                DynamicVars.preferDeltasFromMaster.get()
                        ? SnapshotSource.SLAVE_THEN_DELTAS
                        : SnapshotSource.SLAVE_THEN_MASTER);
    }

    public Option<DataRecord> getRecord(DataApiUserId uid, String handle, RecordId recordId) {
        return dataApiManager.getRecord(consDbSpecWithHandle(uid, handle), recordId);
    }

    public Option<Database> getDatabaseO(DataApiUserId uid) {
        return dataApiManager.getDatabaseO(consDatabaseSpec(uid));
    }

    public Option<Database> getAndInitCopyOnWrite(DataApiUserId uid) {
        return dataApiManager.getAndInitCopyOnWrite(consDatabaseSpec(uid));
    }

    public Option<Database> getIfAlbumsEnoughAndInitCopyOnWrite(DataApiUserId uid, boolean albumsEnabledInMpfs) {
        return getAndInitCopyOnWrite(uid).filter(db -> isAlbumsEnough(db, albumsEnabledInMpfs));
    }

    public Option<Database> getDatabaseIfAlbumsEnough(DataApiUserId uid, boolean albumsEnabledInMpfs) {
        return getDatabaseO(uid).filter(db -> isAlbumsEnough(db, albumsEnabledInMpfs));
    }

    public Option<Database> getDatabaseByHandleO(DataApiUserId uid, String handle) {
        return dataApiManager.getDatabaseO(consDbSpecWithHandle(uid, handle));
    }

    public Database getDatabase(DataApiUserId uid) {
        return dataApiManager.getOrCreateDatabase(consDatabaseSpec(uid));
    }

    public Database createDatabase(DataApiUserId uid) {
        return DynamicVars.photosliceCreateWithAlbums.get()
                ? createDatabaseWithAlbums(uid)
                : dataApiManager.createDatabase(consDatabaseSpec(uid));
    }

    public boolean isAlbumsEnough(Database db, boolean albumsEnabledInMpfs) {
        return AlbumsUtils.isWithAlbums(db) || (!albumsEnabledInMpfs && !DynamicVars.photosliceCreateWithAlbums.get());
    }

    public Database createDatabaseWithAlbums(DataApiUserId uid) {
        return dataApiManager.createDatabaseWithDescription(consDatabaseSpec(uid), AlbumsUtils.ALBUMS_DESCRIPTION);
    }

    private UserDatabaseSpec consDatabaseSpec(DataApiUserId uid) {
        return new UserDatabaseSpec(uid, PHOTOSLICE_DB_REF);
    }

    private Option<SnapshotWithSource> getSnapshotOWithPolicy(DataApiUserId uid, String handle, Option<Long> rev,
            Option<SetF<String>> collectionIdsO, SqlLimits limits, MasterSlavePolicy policy)
    {
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdsO(collectionIdsO)
                .withRecordOrder(DEFAULT_SNAPSHOT_ORDER)
                .withLimits(limits);
        return MasterSlaveContextHolder.withPolicy(policy, () ->
                dataApiManager.getSnapshotWithRevisionO(consDbSpecWithHandle(uid, handle), rev, filter)
        );
    }

    private UserDatabaseSpec consDbSpecWithHandle(DataApiUserId uid, String handle) {
        return UserDatabaseSpec.fromUserAndHandle(uid, PHOTOSLICE_DB_REF.consHandle(handle));
    }

    public Option<SnapshotWithSource> getSnapshotO(DataApiUserId uid, String handle, long rev,
            Option<SetF<String>> collectionIdsO, SqlLimits limits)
    {
        return getSnapshotO(uid, handle, Option.of(rev), collectionIdsO, limits);
    }

    public Option<SnapshotWithSource> getSnapshotO(DataApiUserId uid, String handle, Option<Long> rev,
            Option<SetF<String>> collectionIdsO, SqlLimits limits)
    {
        return getSnapshotOWithPolicy(uid, handle, rev, collectionIdsO, limits, MasterSlavePolicy.R_SM)
                .orElse(() -> getSnapshotOWithPolicy(uid, handle, rev, collectionIdsO, limits, MasterSlavePolicy.R_M));
    }

    public void removeOldDeltas(DataApiUserId uid, String handle, long staringRev, ChunkRateLimiter rateLimiter) {
        getDatabaseByHandleO(uid, handle)
                .forEach(database -> diskDataSource.removeDeltasBefore(staringRev, database, rateLimiter));
    }

    public ListF<Delta> getDeltasList(DataApiUserId uid, String handle, long rev, long dbRev,
            CollectionF<String> collectionId, int maxReturnCount)
    {
        //Can we push down this to Dao level to process the fetched items one-by-one and skip the intermediate list creation?
        Stream<Delta> deltasStream =
                dataApiManager.listDeltas(
                        consDbSpecWithHandle(uid, handle), rev, maxReturnCount)
                        .stream()
                        .filter(d -> d.rev.get() < dbRev)
                        .peek(new DeltaUtils.RevSequenceChecker(rev)::check)
                        .map(Delta::addTargetRevIfMissing);

        if (collectionId.isNotEmpty()) {
            deltasStream = deltasStream.map(delta -> delta.withCollectionChangesOnly(collectionId));
        }
        ListF<Delta> deltaList = Cf.wrap(deltasStream.collect(Collectors.toList()));

        if (deltaList.size() < dbRev - rev && deltaList.size() < maxReturnCount) {
            throw new DeltasGoneException();
        }

        return deltaList;
    }
}
