package ru.yandex.chemodan.app.smartcache.worker.dataapi;

import java.net.UnknownHostException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoCredential;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.DataApiBaseContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.deltas.cleaning.DynamicDeltasCleaningControl;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleaningLagWorkerService;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.DeltaAccessTimeMdao;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.LastRetrievedRevisionTrackerMdao;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.handlers.HandlersContextConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.AutoRateLimiterSupportContextConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mongo3.LoggingCommandListener;
import ru.yandex.commune.mongo3.MonicaCommandListener;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.parse.CommaSeparated;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * @author osidorkin
 */
@Configuration
@Import({
    DataApiBaseContextConfiguration.class,
    HandlersContextConfiguration.class,
    AutoRateLimiterSupportContextConfiguration.class,
})
public class DataApiStorageManagerContextConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(DataApiStorageManagerContextConfiguration.class);

    private final DynamicProperty<Integer> cleanupThreadCount =
            new DynamicProperty<>("smartcache-remove-expired-snapshots-thread-count", 10);

    private final DynamicProperty<Boolean> enableCleanup =
            new DynamicProperty<>("smartcache-cleanup-manager-enable", true);

    private final DynamicProperty<Integer> uidBatchSize =
            new DynamicProperty<>("smartcache-cleanup-manager-uid-batch-size", 100);

    private final DynamicProperty<Integer> maxCleaningUidsByTask =
            new DynamicProperty<>("smartcache-remove-expired-snapshot-max-uids-by-task", 100_000);

    @Bean
    public DataApiStorageManager dataApiStorageManager(DataApiManager dataApiManager,
            DiskDataSource diskDataSource) {
        return new DataApiStorageManager(dataApiManager, diskDataSource);
    }

    @Bean
    public ThreadFactory removeExpiredSnapshotsThreadFactory() {
        return new ThreadNameIndexThreadFactory("smartcache-remove-expired-snapshots");
    }

    @Bean
    public DynamicDeltasCleaningControl dynamicDeltasCleaningControl(
            ShardManager2 dataShardManager,
            UserMetaManager userMetaManager,
            YasmMonitor yasmMonitor,
            MetricsConfiguration metricsConfiguration,
            RateLimitersMetrics rateLimitersMetrics,
            @Value("${sharpei-host-identifier}") String sharpeiIdentifier,
            @Value("${deltas-cleaning.rate-limiter-meter-interval}") Duration meterInterval,
            @Value("${host-status-averaging-interval}") Duration averageInterval,
            @Value("${deltas-cleaning.rate-limiter-maintenance-period}") Duration maintenancePeriod,
            @Value("${dataapi.sharpei.http.max.connections}") int sharpeiMaxConnections)
    {
        return new DynamicDeltasCleaningControl(sharpeiIdentifier,
                dataShardManager, userMetaManager, yasmMonitor, metricsConfiguration, rateLimitersMetrics,
                meterInterval, averageInterval, maintenancePeriod, sharpeiMaxConnections);
    }

    @Bean
    public CleanupManager cleanupManager(DataApiStorageManager dataApiStorageManager,
                                         DeltaAccessTimeMdao deltaAccessTimeMdao,
                                         LastRetrievedRevisionTrackerMdao lastRetrievedRevisionTrackerMdao,
                                         @Value("${smartcache.cleanup-manager.delta-tracking.threads}")
                                                     int deltaTrackingThreadsCount,
                                         @Qualifier("removeExpiredSnapshotsThreadFactory")
                                                     ThreadFactory removeExpiredSnapshotsThreadFactory,
                                         DynamicDeltasCleaningControl dynamicDeltasCleaningControl)
    {
        final ExecutorService deltaTrackingService = Executors.newFixedThreadPool(deltaTrackingThreadsCount);
        return new CleanupManager(dataApiStorageManager, deltaAccessTimeMdao, lastRetrievedRevisionTrackerMdao,
                deltaTrackingService, removeExpiredSnapshotsThreadFactory, cleanupThreadCount::get, enableCleanup::get,
                uidBatchSize::get, dynamicDeltasCleaningControl, maxCleaningUidsByTask::get);
    }

    @Bean
    public DeltaAccessTimeMdao deltaAccessTimeMdao(
                    @Qualifier("expireTrackingMongoClient")
                    MongoClient expireTrackingMongoClient,
                    @Value("${smartcache.expires-tracking.mongo.db.name}")
                    String dbName)
    {
        return new DeltaAccessTimeMdao(expireTrackingMongoClient.getDatabase(dbName));
    }

    @Bean
    public LastRetrievedRevisionTrackerMdao lastRetrievedRevisionTrackerMdao(
                    @Qualifier("expireTrackingMongoClient")
                    MongoClient expireTrackingMongoClient,
                    @Value("${smartcache.expires-tracking.mongo.db.name}")
                    String dbName)
    {
        return new LastRetrievedRevisionTrackerMdao(expireTrackingMongoClient.getDatabase(dbName));
    }

    @Bean
    public MongoClient expireTrackingMongoClient(
            @Value("${smartcache.mongo.hosts:-}")
            GroupOrHost hosts,
            @Value("${smartcache.mongo.port:-27028}")
            IpPort port,
            @Value("${smartcache.mongo.connections.per.host:-50}")
            int connectionsPerHost,
            @Value("${smartcache.mongo.connect.timeout.ms:-5000}")
            int connectTimeoutMs,
            @Value("${smartcache.mongo.socket.timeout.ms:-5000}")
            int socketTimeoutMs,
            @Value("${smartcache.mongo.pool.timeout.ms:-5000}")
            int poolWaitTimeoutMs,
            @Value("${smartcache.mongo.threads.block.multiplier:-20}")
            int threadsAllowedToBlockForConnectionMultiplier,
            @Value("${smartcache.mongo.w:-2}")
            int w,
            @Value("${smartcache.mongo.fsync:-false}")
            boolean fsync,
            @Value("${smartcache.mongo.wtimeout.ms:-2500}")
            int wtimeoutMs,
            @Value("${smartcache.expires-tracking.mongo.db.name}")
            String dbName,
            @Value("${smartcache.mongo.mdb.hosts:-}")
            CommaSeparated mdbHosts,
            @Value("${smartcache.mongo.mdb.user:-}")
            String mdbUser,
            @Value("${smartcache.mongo.mdb.password:-}")
            String mdbPassword,
            @Value("${smartcache.mongo.use.mdb:-false}")
            boolean useMdb,
            Conductor conductor)
            throws UnknownHostException
    {
        ListF<String> mongoHosts = useMdb ? mdbHosts.getList() : conductor.hostsFromString(hosts);
        ListF<ServerAddress> hostList = mongoHosts.map(host -> new ServerAddress(host, port.getPort()));

        MongoClientOptions mongoOptions = MongoClientOptions.builder()
                .connectionsPerHost(connectionsPerHost)
                .connectTimeout(connectTimeoutMs)
                .socketTimeout(socketTimeoutMs)
                .maxWaitTime(poolWaitTimeoutMs)
                .threadsAllowedToBlockForConnectionMultiplier(threadsAllowedToBlockForConnectionMultiplier)
                .writeConcern(new WriteConcern(w, wtimeoutMs, fsync))
                .readPreference(ReadPreference.secondaryPreferred())
                .addCommandListener(new LoggingCommandListener())
                .addCommandListener(new MonicaCommandListener())
                .build();

        logger.info("Starting mongo client to hosts {} with options: {}", hostList,
                YandexToStringBuilder.reflectionToStringValueObject(mongoOptions));


        if (useMdb) {
            MongoCredential credential = MongoCredential.createScramSha1Credential(mdbUser, dbName, mdbPassword.toCharArray());
            return new MongoClient(hostList, credential, mongoOptions);
        }

        return new MongoClient(hostList, mongoOptions);
    }

    @Bean
    public CleaningLagWorkerService cleanupLagWorkerService(DeltaAccessTimeMdao deltaAccessTimeMdao) {
        return new CleaningLagWorkerService(deltaAccessTimeMdao);
    }

}
