package ru.yandex.chemodan.app.smartcache.worker.dataapi;

import java.util.UUID;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestsContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
@ContextConfiguration(classes = {
        TestsContextConfiguration.class
})
@TestExecutionListeners(value = {
        DependencyInjectionTestExecutionListener.class
})
@ActivateDataApiEmbeddedPg
public class DataApiStorageManagerTest extends DataApiTestSupport {

    @Autowired
    private DataApiStorageManager dataApiStorageManager;
    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    private DataApiUserId testUid;

    @Before
    public void before() {
        testUid = createRandomCleanUserInDefaultShard();
        dataApiStorageManager.createDatabaseWithAlbums(testUid);
    }

    @Test
    public void testSnapshotCreateRemoveSimple() {
        testSnapshotCreateRemoveSimple(testUid);
    }

    public void testSnapshotCreateRemoveSimple(DataApiUserId uid) {
        Instant from = new Instant(1417268610);
        Instant to = new Instant(1417324906);
        IndexedCluster cluster = IndexedCluster.consFromLuceneData(
                ClusterId.Format.ADVANCED, from, to, 2, Option.empty(),
                Cf.list(
                        new PhotoViewLuceneInfoPojo("92a4a0d4616df8ceebcd01de06281dea0a32b7c661413f9eab89da7a0b948741",
                                "/disk/Фотокамера/2015-02-04 13-37-15.JPG", new Instant(1423046235000L),
                                Option.empty(), Option.empty(), 1L),
                        new PhotoViewLuceneInfoPojo("0c1a18954071e2c2cdd706335b97efe83658b2bfa9737b8dc8f80f69cf83f0d",
                                "/disk/Фотокамера/1970-01-01 03-00-00_1410979636.JPG", new Instant(1410979609000L),
                                Option.of(42.310478), Option.of(55.640719), 2L))
                );
        ListF<RecordChange> recordChanges = IndexToDeltaMapper.insertClusterToSnapshot(cluster);

        Database indexDatabase = dataApiStorageManager.getDatabase(uid);
        dataApiStorageManager.storeSnapshotIndex(indexDatabase, RevisionCheckMode.WHOLE_DATABASE, recordChanges);
        Snapshot snapshot = dataApiStorageManager.getSnapshot(uid, CollectionIdCondition.all());
        ListF<IndexedCluster> retrievedClustersList = IndexToDeltaMapper.retrieveClusters(snapshot);
        Assert.sizeIs(1, retrievedClustersList);

        IndexedCluster retrieved_cluster = IndexedCluster.consFromLuceneData(
                ClusterId.Format.ADVANCED, from, to, 2, Option.empty(),
                Cf.list(
                        new PhotoViewLuceneInfoPojo("92a4a0d4616df8ceebcd01de06281dea0a32b7c661413f9eab89da7a0b948741",
                                "/disk/Фотокамера/2015-02-04 13-37-15.JPG", new Instant(1423046235000L),
                                Option.empty(), Option.empty(), 1L),
                        new PhotoViewLuceneInfoPojo("0c1a18954071e2c2cdd706335b97efe83658b2bfa9737b8dc8f80f69cf83f0d",
                                "/disk/Фотокамера/1970-01-01 03-00-00_1410979636.JPG", new Instant(1410979609000L),
                                Option.of(42.310478), Option.of(55.640719), 2L))
                );
        Assert.equals(retrieved_cluster.toString(), retrievedClustersList.first().toString());
        dataApiStorageManager.removeDatabase(uid);
    }

    @Test
    public void testSnapshotClustersFiltering() {
        Instant from = new Instant(1417268610);
        Instant to = new Instant(1417324906);
        IndexedCluster cluster = IndexedCluster.consFromLuceneData(
                ClusterId.Format.ADVANCED, from, to, 2, Option.empty(),
                Cf.list(
                        new PhotoViewLuceneInfoPojo("92a4a0d4616df8ceebcd01de06281dea0a32b7c661413f9eab89da7a0b948741",
                                "/disk/Фотокамера/2015-02-04 13-37-15.JPG", new Instant(1423046235000L),
                                Option.empty(), Option.empty(), 1L),
                        new PhotoViewLuceneInfoPojo("0c1a18954071e2c2cdd706335b97efe83658b2bfa9737b8dc8f80f69cf83f0d",
                                "/disk/Фотокамера/1970-01-01 03-00-00_1410979636.JPG", new Instant(1410979609000L),
                                Option.of(42.310478), Option.of(55.640719), 1L))
                );
        ListF<RecordChange> recordChanges = IndexToDeltaMapper.insertClusterToSnapshot(cluster);

        Database indexDatabase = dataApiStorageManager.getDatabase(testUid);
        dataApiStorageManager.storeSnapshotIndex(indexDatabase, RevisionCheckMode.WHOLE_DATABASE, recordChanges);

        Database database = dataApiStorageManager.getDatabaseO(testUid).get();
        Assert.sizeIs(3, getSnapshot(database, Option.empty()).records());

        Option<SetF<String>> collectionIdsO = Option.of(Cf.set(IndexToDeltaMapper.INDEX_COLLECTION_ID));
        Assert.sizeIs(1, getSnapshot(database, collectionIdsO).records());

        dataApiStorageManager.removeDatabase(testUid);
    }

    @Test
    public void testGetBatchedDeltas() {
        dynamicPropertyManager.setValue(DynamicVars.updateSnapshotDiffBatchSize, 2);
        ListF<RecordChange> diffs = Cf.arrayList();
        for (int i = 1; i <= 5; i++) {
            diffs.add(new RecordChange(RecordChangeType.UPDATE, "Collection N " + i, UUID.randomUUID().toString(),
                    Cf.list(FieldChange.delete("Key " + i))));
        }
        int rev = 1;

        ListF<Delta> gotDeltas = dataApiStorageManager.getBatchedDeltas(diffs, rev);

        ListF<Delta> expectedDeltas = Cf.list(
                new Delta(Cf.list(diffs.get(0), diffs.get(1))).withRev(rev),
                new Delta(Cf.list(diffs.get(2), diffs.get(3))).withRev(rev),
                new Delta(Cf.list(diffs.get(4))).withRev(rev)
        );
        org.junit.Assert.assertEquals(expectedDeltas, gotDeltas);
    }

    private Snapshot getSnapshot(Database db, Option<SetF<String>> collectionIdsO) {
        return dataApiStorageManager.getSnapshotO(
                testUid, db.handleValue(), db.rev, collectionIdsO, SqlLimits.all()).get().getSnapshot();
    }
}
