package ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.cleaning.DynamicDeltasCleaningControl;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.ratelimiter.chunk.auto.AutoUserAwareRwRateLimiter;
import ru.yandex.chemodan.app.dataapi.web.ReadonlyException;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.concurrent.ExecutorUtils;
import ru.yandex.chemodan.ratelimiter.NegativeRateException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.auto.HostKey;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author osidorkin
 */
public class CleanupManager {
    private static final Logger logger = LoggerFactory.getLogger(CleanupManager.class);
    private static final int RETRY_COUNT = 3;

    private final DeltaAccessTimeMdao deltaAccessTimeMdao;
    private final LastRetrievedRevisionTrackerMdao lastRetrievedRevisionTrackerMdao;
    private final DataApiStorageManager dataApiStorageManager;
    private final ExecutorService deltaTrackingService;
    private final ThreadFactory cleanupThreadFactory;
    private final Supplier<Integer> threadCountProvider;
    private final Supplier<Boolean> enableCleanupProvider;
    private final Supplier<Integer> uidsBatchSizeProvider;
    private final AutoUserAwareRwRateLimiter userAwareRateLimiter;
    private final Supplier<Integer> maxCleaningUidsByTaskProvider;

    public CleanupManager(DataApiStorageManager dataApiStorageManager, DeltaAccessTimeMdao deltaAccessTimeMdao,
                          LastRetrievedRevisionTrackerMdao lastRetrievedRevisionTrackerMdao,
                          ExecutorService deltaTrackingService, ThreadFactory cleanupThreadFactory,
                          Supplier<Integer> threadCountProvider, Supplier<Boolean> enableCleanupProvider,
                          Supplier<Integer> uidsBatchSizeProvider,
                          DynamicDeltasCleaningControl dynamicDeltasCleaningControl,
                          Supplier<Integer> maxCleaningUidsByTaskProvider)
    {
        this.deltaAccessTimeMdao = deltaAccessTimeMdao;
        this.lastRetrievedRevisionTrackerMdao = lastRetrievedRevisionTrackerMdao;
        this.dataApiStorageManager = dataApiStorageManager;
        this.deltaTrackingService = deltaTrackingService;
        this.cleanupThreadFactory = cleanupThreadFactory;
        this.threadCountProvider = threadCountProvider;
        this.enableCleanupProvider = enableCleanupProvider;
        this.uidsBatchSizeProvider = uidsBatchSizeProvider;
        this.userAwareRateLimiter = dynamicDeltasCleaningControl.getAutoRateLimiter();
        this.maxCleaningUidsByTaskProvider = maxCleaningUidsByTaskProvider;
    }

    public void registerCreatedDatabase(Database database) {
        revisionRetrieved(database.uid, database.handleValue(), database.rev);
        startTrackingDelta(database.uid, database.handleValue(), database.rev);
    }

    public void unregisterDatabase(Database database) {
        deltaAccessTimeMdao.removeTracking(database.uid, database.handleValue());
        lastRetrievedRevisionTrackerMdao.untrackRevision(database.uid, database.handleValue());
    }

    public void startTrackingDelta(DataApiUserId uid, String handle, long revision) {
        deltaAccessTimeMdao.startTracking(uid, handle, revision);
    }

    // Not the same as revisionRetrieved since we are tracking deltas access only here
    public void deltaRetrieved(DataApiUserId uid, String handle, long minRevision, long snapshotRevision) {
        executeAsync(() -> deltaRetrievedTask(uid, handle, minRevision, snapshotRevision));
    }

    private void deltaRetrievedTask(DataApiUserId uid, String handle, long minRevision, long snapshotRevision) {
        Option<TrackedDelta> trackedDeltaO = deltaAccessTimeMdao.getTrackedItem(uid, handle);
        if (!trackedDeltaO.isPresent()) {
            startTrackingDelta(uid, handle, snapshotRevision);
            return;
        }
        if (trackedDeltaO.get().revision >= minRevision) {
            deltaAccessTimeMdao.recordAccess(uid, handle);
        }
    }

    public void revisionRetrieved(DataApiUserId uid, String handle, long revision) {
        executeAsync(() -> revisionRetrievedTask(uid, handle, revision));
    }

    private void revisionRetrievedTask(DataApiUserId uid, String handle, long revision) {
        Option<Long> trackedRevisionO = lastRetrievedRevisionTrackerMdao.getCurrentRevision(uid, handle);
        if (!trackedRevisionO.isPresent() || trackedRevisionO.get().longValue() < revision) {
            lastRetrievedRevisionTrackerMdao.updateRevision(uid, handle, revision);
        }
    }

    public CleanupStats removeExpired() {
        return removeExpired(DefaultDeltaCleanupData::create, this::processTrackedDelta, Option.empty());
    }

    public CleanupStats removeExpiredInParallel() {
        int threadCount = threadCountProvider.get();
        if (threadCount < 1) {
            return new CleanupStats(0, 0);
        }
        ExecutorService executor = Executors.newFixedThreadPool(threadCount, cleanupThreadFactory);
        Semaphore semaphore = new Semaphore(threadCount);
        try {
            CleanupStats result = removeExpired(DefaultDeltaCleanupData::createConcurrent,
                    (delta, deltaCleanupData) -> processTrackedDeltaInParallel(delta, deltaCleanupData, semaphore, executor),
                    Option.of(() -> waitForCleanupProcessors(semaphore, threadCount)));
            executor.shutdown();
            return result;
        } catch (Exception e) {
            executor.shutdownNow();
            throw e;
        }
    }

    public void waitForCleanupProcessors(Semaphore semaphore, int threadCount) {
        while (semaphore.availablePermits() < threadCount) {
            ThreadUtils.doSleep(10, TimeUnit.MILLISECONDS);
        }
    }

    public void processTrackedDeltaInParallel(TrackedDelta delta, DeltaCleanupData deltaCleanupData,
                                              Semaphore semaphore, ExecutorService executorService)
    {
        try {
            semaphore.acquire();
        } catch (InterruptedException e) {
            logger.debug(e);
            throw ExceptionUtils.translate(e);
        }

        ExecutorUtils.submitWithYcridForwarding(() -> {
            try {
                processTrackedDelta(delta, deltaCleanupData);
            } finally {
                semaphore.release();
            }
        }, executorService);
    }

    public CleanupStats removeExpired(Supplier<? extends DeltaCleanupData> deltaCleanupDataProvider,
                                      BiConsumer<? super TrackedDelta, ? super DeltaCleanupData> deltaCleanupProcessor,
                                      Option<Function0V> afterBatchCleanupProcessor)
    {
        Instant expirationInstant = getExpirationInstant();
        int maxCleanupUids = maxCleaningUidsByTaskProvider.get();
        int cleanupUidsCount = 0;
        DeltaCleanupData deltaCleanupData = deltaCleanupDataProvider.get();
        ListF<TrackedDelta> tracked;

        while (cleanupUidsCount < maxCleanupUids &&
                (tracked = deltaAccessTimeMdao.getTrackedItemsBefore(expirationInstant, uidsBatchSizeProvider.get())).isNotEmpty()) {

            if (!enableCleanupProvider.get()) {
                break;
            }

            CleanupMetrics.mongoGetDeltasToDelete.inc(tracked.size());
            tracked.forEach(delta -> deltaCleanupProcessor.accept(delta, deltaCleanupData));
            afterBatchCleanupProcessor.ifPresent(Function0V::apply);
            cleanupUidsCount += tracked.size();
        }
        List<DataApiUserId> failedUids = deltaCleanupData.getFailedUids();
        int removedSnapshotsCount = deltaCleanupData.getRemovedSnapshotsCount();

        if (failedUids.size() > 0) {
            // Не должно никогда вызываться, но пусть пока останется
            throw new IllegalStateException(String.format(
                    "%s deltas has not been deleted successfully. cleanupUids=%s removedSnapshotsCount=%s failedUids=%s",
                    failedUids.size(), cleanupUidsCount, removedSnapshotsCount, failedUids
            ));
        }
        return new CleanupStats(cleanupUidsCount, removedSnapshotsCount);
    }

    public void removeExpiredDelta(DataApiUserId uid, String handle) {
        Option<TrackedDelta> trackedDeltaO = deltaAccessTimeMdao.getTrackedItem(uid, handle);
        trackedDeltaO.ifPresent(trackedDelta -> {
            long expirationTime = getExpirationInstant().getMillis();
            if (trackedDelta.lastAccessed > expirationTime) {
                logger.info("Last access time for delta is less that expiration time: uid={} handle={} rev={}",
                        trackedDelta.deltaId.uid, trackedDelta.deltaId.handle, trackedDelta.revision);
                return;
            }

            processTrackedDelta(trackedDelta);
        });
    }

    private void processTrackedDelta(TrackedDelta delta, DeltaCleanupData deltaCleanupData) {
        Either<Integer, Throwable> result = removeExpiredTrackedDelta(delta);
        if (result.isLeft()) {
            deltaCleanupData.addRemovedSnapshotCount(result.getLeft());
            deltaCleanupData.removeFailedUid(delta.deltaId.uid);
            return;
        }

        if (result.getRight() instanceof ReadonlyException) {
            logger.info("User {} is readonly, skipping removeExpiredDeltas", delta.deltaId.uid);
            return;
        }
        deltaCleanupData.addFailedUid(delta.deltaId.uid);
    }

    private void processTrackedDelta(TrackedDelta delta) {
        Either<Integer, Throwable> result = removeExpiredTrackedDelta(delta);
        if (result.isRight()) {
            ExceptionUtils.throwException(result.getRight());
        }
    }

    private Either<Integer, Throwable> removeExpiredTrackedDelta(TrackedDelta delta) {
        HostKey hostKey = userAwareRateLimiter.getHostKey(delta.deltaId.uid);
        String shard = String.valueOf(hostKey.getShardId());

        Either<Integer, Throwable> result =
                RetryUtils.retryE(logger, RETRY_COUNT, () -> removeExpiredDeltas(delta, hostKey),
                        e -> e instanceof NegativeRateException);
        if (result.isLeft()) {
            CleanupMetrics.successDeleteByShard.inc(shard);
            return result;
        }

        CleanupMetrics.exceptionsByShard.inc(result.getRight().getClass().getSimpleName(), shard);
        return result;
    }

    private int removeExpiredDeltas(TrackedDelta trackedDelta, HostKey hostKey) {
        ChunkRateLimiter rateLimiter = userAwareRateLimiter.getWriteRateLimiter(hostKey);
        String shard = String.valueOf(hostKey.getShardId());

        MongoDeltaId id = trackedDelta.deltaId;
        dataApiStorageManager.removeOldDeltas(id.uid, id.handle, trackedDelta.revision, rateLimiter);
        CleanupMetrics.postgresqlDeleteRequestByShard.inc(shard);

        Option<Long> revisionToSetO = getLastRetrievedRevision(id.uid, id.handle);
        if (!revisionToSetO.isPresent()) {
            deltaAccessTimeMdao.removeTracking(id.uid, id.handle);
            return 0;
        }

        if (revisionToSetO.get() == trackedDelta.revision) {
            dataApiStorageManager.removeDatabaseByHandle(id.uid, id.handle, rateLimiter);
            deltaAccessTimeMdao.removeTracking(id.uid, id.handle);
            lastRetrievedRevisionTrackerMdao.untrackRevision(id.uid, id.handle);
            CleanupMetrics.postgresqlDropAllDeltasByShard.inc(shard);
            return 1;
        }

        deltaAccessTimeMdao.startTracking(id.uid, id.handle, revisionToSetO.get());
        return 0;
    }

    private Option<Long> getLastRetrievedRevision(DataApiUserId uid, String handle) {
        Option<Long> currentRetrievedRevisionO = lastRetrievedRevisionTrackerMdao.getCurrentRevision(uid, handle);
        if (currentRetrievedRevisionO.isPresent()) {
            return currentRetrievedRevisionO;
        }
        Option<Database> databaseO = dataApiStorageManager.getDatabaseByHandleO(uid, handle);
        if (!databaseO.isPresent()) {
            return Option.empty();
        }
        lastRetrievedRevisionTrackerMdao.updateRevision(uid, handle, databaseO.get().rev);
        return Option.of(databaseO.get().rev);
    }

    private void executeAsync(Runnable task) {
        ExecutorUtils.submitWithYcridForwarding(task, deltaTrackingService);
    }

    public static Instant getExpirationInstant() {
        int expirationDays = DynamicVars.photosliceDeltaExpirationDays.get();
        return Instant.now().minus(Duration.standardDays(expirationDays));
    }
}
