
package ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup;

import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import com.mongodb.BasicDBObject;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.deltas.cleaning.DynamicDeltasCleaningControl;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestsContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author osidorkin
 */
@ContextConfiguration(classes = {
        TestsContextConfiguration.class
})
@TestExecutionListeners(value = {
        DependencyInjectionTestExecutionListener.class
})
@YaIgnore
public class CleanupManagerTest extends DataApiTestSupport {

    @Autowired
    private DataApiStorageManager storageManager;
    @Autowired
    private DeltaAccessTimeMdao deltaAccessTimeMdao;
    @Autowired
    private LastRetrievedRevisionTrackerMdao lastRetrievedRevisionTrackerMdao;
    @Autowired
    private DynamicDeltasCleaningControl dynamicDeltasCleaningControl;

    private CleanupManager cleanupManager;

    @Before
    public void before() {
        cleanupMongo();
        cleanupManager = new CleanupManager(storageManager, deltaAccessTimeMdao, lastRetrievedRevisionTrackerMdao,
                Executors.newFixedThreadPool(1), Mockito.mock(ThreadFactory.class), () -> 1, () -> true, () -> 100,
                dynamicDeltasCleaningControl, () -> 10000);
    }

    @After
    public void after() {
        cleanupMongo();
    }

    @Test
    public void testCleanup() {
        final int expirationDays = DynamicVars.photosliceDeltaExpirationDays.get();
        Instant oldInstant = Instant.now().minus(Duration.standardDays(expirationDays * 2));

        DataApiUserId newUid = createRandomCleanUserInDefaultShard();
        DataApiUserId cleanupUid = createRandomCleanUserInDefaultShard();
        DataApiUserId removeUid = createRandomCleanUserInDefaultShard();

        lastRetrievedRevisionTrackerMdao.updateRevision(newUid, newUid.toString(), 100);
        lastRetrievedRevisionTrackerMdao.updateRevision(cleanupUid, cleanupUid.toString(), 100);
        lastRetrievedRevisionTrackerMdao.updateRevision(removeUid, removeUid.toString(), 100);

        deltaAccessTimeMdao.startTracking(new TrackedDelta(id(newUid) , 50, oldInstant.getMillis()));
        deltaAccessTimeMdao.startTracking(new TrackedDelta(id(cleanupUid) , 50, oldInstant.getMillis()));
        deltaAccessTimeMdao.startTracking(new TrackedDelta(id(removeUid) , 100, oldInstant.getMillis()));

        ListF<TrackedDelta> currentRecodedDeltas = deltaAccessTimeMdao.getTrackedItemsBefore(Instant.now(), 100);
        Assert.sizeIs(3, currentRecodedDeltas);

        cleanupManager.deltaRetrieved(newUid, newUid.toString(), 50, 100);
        ThreadUtils.sleep(100);
        CleanupStats stats = cleanupManager.removeExpired();
        Assert.equals(2, stats.uidsCleanuped);
        Assert.equals(1, stats.snapshotsRemoved);

        currentRecodedDeltas = deltaAccessTimeMdao.getTrackedItemsBefore(Instant.now(), 100);
        Assert.sizeIs(2, currentRecodedDeltas);

        stats = cleanupManager.removeExpired();
        Assert.equals(0, stats.uidsCleanuped);
        Assert.equals(0, stats.snapshotsRemoved);
    }

    private void cleanupMongo() {
        deltaAccessTimeMdao.getCollection().deleteMany(new BasicDBObject());
        lastRetrievedRevisionTrackerMdao.getCollection().deleteMany(new BasicDBObject());
    }

    private static MongoDeltaId id(DataApiUserId uid) {
        return new MongoDeltaId(uid, uid.toString());
    }

}
