package ru.yandex.chemodan.app.smartcache.worker.dataapi.handlers.async;

import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventAsyncHandler;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaUtils;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.xiva.DiskLikeXivaPushBody;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaPushBodyRepack;
import ru.yandex.chemodan.xiva.XivaPushService;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.MarshallerContext;

/**
 * @author Denis Bakharev
 * @author Dmitriy Amelin (lemeh)
 */
public class XivaPushSender implements DatabaseChangedEventAsyncHandler {

    public static final String SMARTCACHE_XIVA_SERVICE = "disk-json";

    private static final String PHOTOSLICE_UPDATED_EVENT = "photoslice_updated";

    private final XivaSingleTokenClient xivaClient;

    public XivaPushSender(XivaSingleTokenClient xivaClient) {
        this.xivaClient = xivaClient;
    }

    @Override
    public void databaseChanged(DatabaseChange databaseChange) {
        Database database = databaseChange.patchedDatabase();
        if (isPhotoslice(database)) {
            sendPushAboutNewPhotoslice(database);
        }
    }

    private boolean isPhotoslice(Database database) {
        return DataApiStorageManager.PHOTOSLICE_DB_REF.refersDbMatching(database);
    }

    private void sendPushAboutNewPhotoslice(Database database) {
        xivaClient.send(
                new XivaEvent(DataApiXivaUtils.toXivaRecipient(database.uid), PHOTOSLICE_UPDATED_EVENT)
                        .withBody(new PhotoslicePushBody(database))
        );
    }

    @Bendable
    static class PhotoslicePushBody extends DiskLikeXivaPushBody {
        @BenderPart(name = "parameters", strictName = true)
        final Params params;

        final String collapseKey;

        PhotoslicePushBody(Database database) {
            super(PHOTOSLICE_UPDATED_EVENT);
            this.params = new Params(database);
            this.collapseKey = "photoslice_updated_" + database.uid.toString();
        }

        @Bendable
        private static class Params {
            @BenderPart(name = "photoslice_id", strictName = true)
            final String photosliceId;

            @BenderPart(name = "current_revision", strictName = true)
            final long currentRevision;

            Params(Database database) {
                this.photosliceId = database.handleValue();
                this.currentRevision = database.rev;
            }
        }

        @Override
        public XivaPushBodyRepack getRepack() {
            return new XivaPushBodyRepack()
                    .withServiceRepack(XivaPushService.GCM, repack -> repack.addStandardField("collapse_key", collapseKey))
                    .withServiceRepack(XivaPushService.APNS)
                    .includeSpecialAndAllPayloadFieldsToAllServices();
        }

        @Override
        public void writePushBodyAsJson(BenderJsonWriter writer, MarshallerContext context) {
            mapper.serializeJson(new Wrapper(this), writer, context);
        }

        @Bendable
        protected static class Wrapper {
            @BenderPart(name = "root")
            final DiskLikeXivaPushBody content;

            @BenderPart(name = "t", strictName = true)
            final String tag = "photoslice_updated";

            @BenderPart(name = "p_id", strictName = true)
            final String photosliceId;

            @BenderPart(name = "p_r", strictName = true)
            final long currentRevision;

            public Wrapper(PhotoslicePushBody content) {
                this.content = content;
                this.photosliceId = content.params.photosliceId;
                this.currentRevision = content.params.currentRevision;
            }
        }

    }
}
