package ru.yandex.chemodan.app.smartcache.worker.dataapi.handlers.async;

import java.util.List;

import com.xebialabs.restito.semantics.Call;
import org.glassfish.grizzly.http.Method;
import org.glassfish.grizzly.http.util.HttpStatus;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiRandomValueGenerator;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.util.JsonAssert;
import ru.yandex.chemodan.util.test.StubServerUtils;
import ru.yandex.chemodan.xiva.BasicXivaClient;
import ru.yandex.chemodan.xiva.XivaClient;
import ru.yandex.chemodan.xiva.XivaTestUtils;
import ru.yandex.commune.json.JsonUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.IpPortUtils;
import ru.yandex.misc.test.Assert;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.status;
import static com.xebialabs.restito.semantics.Condition.method;

/**
 * @author Denis Bakharev
 */
public class XivaPushSenderTest {

    @Test
    public void databaseChanged_WithNewRevision_MustSentPush() {
        int port = IpPortUtils.getFreeLocalPort().getPort();

        BasicXivaClient basicXivaClient = new BasicXivaClient(
                "http://localhost:" + port + "/",
                ApacheHttpClientUtils.Builder.create()
                        .singleThreaded()
                        .withMaxConnections(1)
                        .withRequestRetry(0, false)
                        .build()
        );
        MapF<String, String> tokens = Cf.map("test", "123");
        XivaClient client = new XivaClient(basicXivaClient, tokens);
        XivaPushSender xivaPushSender = new XivaPushSender(client.toSingleTokenClient("test"));

        StubServerUtils.withStubServer(port, stubServer -> {
            whenHttp(stubServer).match(method(Method.POST)).then(status(HttpStatus.OK_200));

            DataApiPassportUserId uid = new DataApiPassportUserId(101);
            Database database = new Database(
                    uid,
                    DataApiStorageManager.PHOTOSLICE_DB_REF
                            .consHandle("handle"),
                    100,
                    new DatabaseMeta(Instant.now(), Instant.now(), DataSize.fromBytes(1), 0));
            xivaPushSender.databaseChanged(DatabaseChange.empty(database));

            List<Call> calls = stubServer.getCalls();
            Assert.equals(1, calls.size());

            String expectedBody =
                    "{" +
                            "\"payload\":{" +
                                "\"root\":{" +
                                    "\"tag\":\"photoslice_updated\"," +
                                    "\"parameters\":{" +
                                        "\"photoslice_id\":\"handle\"," +
                                        "\"current_revision\":100" +
                                    "}" +
                                "}," +
                                "\"t\":\"photoslice_updated\"," +
                                "\"p_id\":\"handle\"," +
                                "\"p_r\":100" +
                            "}," +
                            "\"repack\":{" +
                                "\"gcm\":{" +
                                    "\"collapse_key\":\"photoslice_updated_101\"," +
                                    "\"repack_payload\":[" +
                                        "\"root\"," +
                                        "\"t\"," +
                                        "\"p_id\"," +
                                        "\"p_r\"," +
                                        "{" +
                                            "\"r\":\"::xiva::push_token\"" +
                                        "}," +
                                        "{" +
                                            "\"transit_id\":\"::xiva::transit_id\"" +
                                        "}" +
                                    "]" +
                                "}," +
                                "\"apns\":{" +
                                    "\"repack_payload\":[" +
                                        "\"root\"," +
                                        "\"t\"," +
                                        "\"p_id\"," +
                                        "\"p_r\"," +
                                        "{" +
                                            "\"r\":\"::xiva::push_token\"" +
                                        "}," +
                                        "{" +
                                            "\"transit_id\":\"::xiva::transit_id\"" +
                                        "}" +
                                    "]" +
                                "}," +
                                "\"other\":{" +
                                    "\"repack_payload\":[" +
                                        "\"root\"," +
                                        "\"t\"," +
                                        "\"p_id\"," +
                                        "\"p_r\"," +
                                        "{" +
                                            "\"r\":\"::xiva::push_token\"" +
                                        "}," +
                                        "{" +
                                            "\"transit_id\":\"::xiva::transit_id\"" +
                                        "}" +
                                    "]" +
                                "}" +
                            "}" +
                    "}";

            JsonAssert.equals(expectedBody, calls.get(0).getPostBody());

            assertContainsParameter(calls.get(0), "event", "photoslice_updated");
            assertContainsParameter(calls.get(0), "uid", "101");

        });
    }

    private void assertContainsParameter(Call call, String expectedParameter, String expectedValue) {
        Assert.equals(expectedValue, call.getParameters().get(expectedParameter)[0]);
    }

    @Test
    public void pushBody() {
        Database database = new DataApiRandomValueGenerator().createDatabase();
        JsonAssert.equals(
                ("{\n"
                + "    \"root\": {\n"
                + "        \"tag\": \"photoslice_updated\",\n"
                + "        \"parameters\": {\n"
                + "            \"photoslice_id\": " + JsonUtils.quoteJson(database.handleValue()) + ",\n"
                + "            \"current_revision\": " + database.rev + "\n"
                + "        }\n"
                + "    },\n"
                + "    \"t\":\"photoslice_updated\",\n"
                + "    \"p_id\":" + JsonUtils.quoteJson(database.handleValue()) + ",\n"
                + "    \"p_r\":" + database.rev + "\n"
                + "}"),
                XivaTestUtils.getSerializedForm(new XivaPushSender.PhotoslicePushBody(database))
        );
    }
}
