package ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers;

import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.support.RecordFieldUtils;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.AlbumMetaPojo;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author yashunsky
 */
public class AlbumsToDeltaMapper {
    public static final String ALBUMS_COLLECTION_ID = "_albums";

    private static final int ALBUMS_PREVIEWS_COUNT = 10;

    public static AlbumType parseAlbumTypeFromField(String key) {
        return AlbumType.valueOf(StringUtils.substringAfter(key, ClusterKeys.ALBUM_PREFIX).toUpperCase());
    }

    public static AlbumType parseAlbumTypeFromRecordId(String key) {
        return AlbumType.valueOf(key.toUpperCase());
    }

    public static String albumTypeToRecordId(AlbumType albumType) {
        return albumType.name().toLowerCase();
    }

    public static ListF<RecordChange> getAlbumsChanges(
            ListF<AlbumMetaPojo> albumsData, ListF<PhotoViewLuceneClusterPojo> clusterPojos) {
        MapF<AlbumType, AlbumMetaPojo> oldAlbums = albumsData.toMap(AlbumMetaPojo::getType, data -> data);
        MapF<AlbumType, AlbumMetaPojo> newAlbums = getAlbumsMeta(clusterPojos);

        return Cf.x(AlbumType.values()).filterMap(
                album -> getRecordChange(oldAlbums.getO(album), newAlbums.getO(album)));
    }

    private static MapF<AlbumType, AlbumMetaPojo> getAlbumsMeta(ListF<PhotoViewLuceneClusterPojo> clusterPojos) {
        ListF<AlbumType> albums = Cf.x(AlbumType.values());

        MapF<AlbumType, ListF<String>> albumPreviews =
                albums.zipWith(t -> Cf.<String>arrayListWithCapacity(ALBUMS_PREVIEWS_COUNT)).toMap();
        MapF<AlbumType, AtomicInteger> albumCounters = albums.zipWith(t -> new AtomicInteger(0)).toMap();

        ListF<PhotoViewLuceneInfoPojo> allPhotos =
                clusterPojos.sortedByDesc(cluster -> cluster.max).flatMap(cluster -> cluster.mergedDocs);

        for (PhotoViewLuceneInfoPojo photo : allPhotos) {
            for (AlbumType album : photo.getAlbums()) {
                albumCounters.getTs(album).incrementAndGet();
                ListF<String> previews = albumPreviews.getTs(album);
                if (previews.size() < ALBUMS_PREVIEWS_COUNT) {
                    previews.add(photo.key);
                }
            }
        }

        return albums.zipWith(album ->
                new AlbumMetaPojo(album, albumPreviews.getTs(album), albumCounters.getTs(album).get()))
                .toMap().filterValues(AlbumMetaPojo::isNotEmpty);
    }

    private static Option<RecordChange> getRecordChange(Option<AlbumMetaPojo> oldData, Option<AlbumMetaPojo> newData) {

        if (oldData.isPresent() && newData.isPresent()) {
            return getUpdateRecordChange(oldData.get(), newData.get());
        } else if (oldData.isPresent()) {
            return Option.of(getDeleteRecordChange(oldData.get()));
        } else if (newData.isPresent()) {
            return Option.of(getInsertRecordChange(newData.get()));
        }
        return Option.empty();
    }

    private static Option<RecordChange> getUpdateRecordChange(AlbumMetaPojo oldData, AlbumMetaPojo newData) {
        Validate.equals(oldData.getType(), newData.getType());

        MapF<String, DataField> oldDataFields = collectFields(oldData);
        MapF<String, DataField> newDataFields = collectFields(newData);

        ListF<FieldChange> changes = RecordFieldUtils.diff(oldDataFields, newDataFields);

        if (changes.isEmpty()) {
            return Option.empty();
        }

        return Option.of(RecordChange.update(ALBUMS_COLLECTION_ID, albumTypeToRecordId(newData.getType()), changes));
    }

    private static RecordChange getInsertRecordChange(AlbumMetaPojo data) {
        return RecordChange.insert(ALBUMS_COLLECTION_ID, albumTypeToRecordId(data.getType()), collectFields(data));
    }

     private static RecordChange getDeleteRecordChange(AlbumMetaPojo data) {
        return RecordChange.delete(ALBUMS_COLLECTION_ID, albumTypeToRecordId(data.getType()));
    }

    private static MapF<String, DataField> collectFields(AlbumMetaPojo meta) {
        MapF<String, DataField> result = Cf.linkedHashMap();

        result.put(AlbumFields.PREVIEWS.toData(meta.getPreviews()));
        result.put(AlbumFields.COUNT.toData(meta.getCount()));

        return result;
    }

    public static AlbumMetaPojo parseDataRecord(DataRecord record) {
        Validate.equals(ALBUMS_COLLECTION_ID, record.getCollectionId());
        return new AlbumMetaPojo(parseAlbumTypeFromRecordId(record.getRecordId()),
                AlbumFields.PREVIEWS.get(record), AlbumFields.COUNT.get(record));
    }

}
