package ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function1I;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.app.smartcache.worker.utils.ListDiffCalculator;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author osidorkin
 */
enum IndexFieldsSerializer {

    FROM_INSTANT(ClusterKeys.FROM_INSTANT, IndexedCluster::getFrom),
    TO_INSTANT(ClusterKeys.TO_INSTANT, IndexedCluster::getTo),
    PHOTOS_COUNT(ClusterKeys.PHOTOS_COUNT, Function1B.trueF(), equalsIntF(IndexedCluster::getPhotosCount),
            integerFieldF(IndexedCluster::getPhotosCount)),
    PHOTOS_HASH(ClusterKeys.PHOTOS_HASH, IndexedCluster::getPhotosHash, DataField::string),
    LOCALITY(ClusterKeys.LOCALITY, IndexedCluster::getLocalityO,
            IndexFieldsSerializer::localizedStringDictionaryToField),
    PLACES(ClusterKeys.PLACES, combineBF(ListF::isNotEmpty, IndexedCluster::getPlaces),
            equalsF(IndexedCluster::getPlaces),
            combineF((values) -> DataField.list(values.map(IndexToDeltaMapper::localizedStringDictionaryToField)),
                    IndexedCluster::getPlaces), new LocationsFieldUpdater()),

    ALBUM_CAMERA(AlbumType.CAMERA),
    ALBUM_SCREENSHOTS(AlbumType.SCREENSHOTS),
    ALBUM_VIDEOS(AlbumType.VIDEOS),
    ALBUM_BEAUTIFUL(AlbumType.BEAUTIFUL),
    ALBUM_UNBEAUTIFUL(AlbumType.UNBEAUTIFUL),
    ALBUM_NONPHOTOUNLIM(AlbumType.NONPHOTOUNLIM),
    ALBUM_PHOTOUNLIM(AlbumType.PHOTOUNLIM);

    private final String key;
    private final Function1B<IndexedCluster> fieldExistsF;
    private final Function2B<IndexedCluster, IndexedCluster> fieldEqualsF;
    private final Function<IndexedCluster, DataField> toDataFieldMapperF;
    private final FieldsUpdateCalculator fieldsUpdateCalculator;

    public static String getAlbumFieldName(AlbumType type) {
        return ClusterKeys.ALBUM_PREFIX + StringUtils.capitalize(type.name().toLowerCase());
    }

    //Album fields
    private <T> IndexFieldsSerializer(AlbumType type) {
        this(getAlbumFieldName(type),
                cluster -> cluster.hasPhotosInAlbum(type),
                equalsIntF(cluster -> cluster.getPhotosInAlbumCount(type)),
                integerFieldF(cluster -> cluster.getPhotosInAlbumCount(type)));
    }

    //Timestamp fields
    private <T> IndexFieldsSerializer(String key, Function<IndexedCluster, Instant> valueExtractorF) {
        this(key, Function1B.trueF(), equalsF(valueExtractorF), combineF(DataField::timestamp, valueExtractorF));
    }

    //Option fields
    private <T> IndexFieldsSerializer(String key, Function<IndexedCluster, Option<T>> valueExtractorF,
            Function<T, DataField> dataFieldMapperF)
    {
        this(key, combineBF(Option::isDefined, valueExtractorF), equalsF(valueExtractorF),
                combineF(dataFieldMapperF, getOptionValueF(valueExtractorF)));
    }


    private IndexFieldsSerializer(String key, Function1B<IndexedCluster> fieldExistsF,
            Function2B<IndexedCluster, IndexedCluster> fieldEqualsF, Function<IndexedCluster, DataField> toDataFieldMapperF)
    {
        this(key, fieldExistsF, fieldEqualsF, toDataFieldMapperF, PlainFieldsUpdater.PLAIN_FIELD_UPDATER);
    }

    private IndexFieldsSerializer(String key, Function1B<IndexedCluster> fieldExistsF,
            Function2B<IndexedCluster, IndexedCluster> fieldEqualsF, Function<IndexedCluster, DataField> toDataFieldMapperF,
            FieldsUpdateCalculator fieldsUpdateCalculator)
    {
        this.key = key;
        this.fieldExistsF = fieldExistsF;
        this.fieldEqualsF = fieldEqualsF;
        this.toDataFieldMapperF = toDataFieldMapperF;
        this.fieldsUpdateCalculator = fieldsUpdateCalculator;
    }

    private void addField(MapF<String, DataField> fields, IndexedCluster cluster) {
        if (!fieldExistsF.apply(cluster)) {
            return;
        }
        fields.put(key, toDataFieldMapperF.apply(cluster));
    }

    private void addFieldDiff(ListF<FieldChange> changes, IndexedCluster oldCluster, IndexedCluster newCluster) {
        if (fieldEqualsF.apply(oldCluster, newCluster)) {
            return;
        }
        if (!fieldsUpdateCalculator.isSimpleDeletion() || fieldExistsF.apply(newCluster)) {
            fieldsUpdateCalculator.apply(oldCluster, newCluster, this, changes);
        } else {
            changes.add(FieldChange.delete(key));
        }
    }

    static DataField localizedStringDictionaryToField(LocalizedStringDictionary dict) {
        return DataField.list(dict.toList().map(DataField::string));
    }

    static DataField albumsToField(MapF<AlbumType, Integer> albumEntries) {
        return DataField.map(albumEntries.entries().map1(Enum::toString).map2(DataField::integer).toMap());
    }

    public static MapF<String, DataField> indexedClusterToFields(IndexedCluster indexedCluster) {
        MapF<String, DataField> res = Cf.hashMap();
        for (IndexFieldsSerializer mapper: values()) {
            mapper.addField(res, indexedCluster);
        }
        return res;
    }

    public static ListF<FieldChange> indexedClusterDiffToFields(IndexedCluster oldCluster, IndexedCluster newCluster) {
        ListF<FieldChange> res = Cf.arrayList();
        for (IndexFieldsSerializer mapper: values()) {
            mapper.addFieldDiff(res, oldCluster, newCluster);
        }
        return res;
    }

    private static <V> Function<IndexedCluster, V> getOptionValueF(Function<IndexedCluster, Option<V>> mapperF) {
        return (c) -> mapperF.apply(c).get();
    }

    private static <T> Function1B<IndexedCluster> combineBF(Function1B<T> externalF, Function<IndexedCluster, T> mapperF) {
        return (c) -> externalF.apply(mapperF.apply(c));
    }

    private static <T, U, V> Function<U, V> combineF(Function<T, V> externalF, Function<U, T> internalF) {
        return (c) -> externalF.apply(internalF.apply(c));
    }

    private static <T> Function<IndexedCluster, DataField> integerFieldF(Function1I<IndexedCluster> mapperF) {
        return (c) -> DataField.integer(mapperF.apply(c));
    }

    private static <T> Function2B<IndexedCluster, IndexedCluster> equalsIntF(Function1I<IndexedCluster> mapperF) {
        return (c1, c2) -> mapperF.apply(c1) == (mapperF.apply(c2));
    }

    private static <T> Function2B<IndexedCluster, IndexedCluster> equalsF(Function<IndexedCluster, T> mapperF) {
        return (c1, c2) -> mapperF.apply(c1).equals(mapperF.apply(c2));
    }

    private interface FieldsUpdateCalculator {
        void apply(IndexedCluster oldCluster, IndexedCluster newCluster, IndexFieldsSerializer instance, ListF<FieldChange> destination);
        boolean isSimpleDeletion();
    }

    private static class PlainFieldsUpdater implements FieldsUpdateCalculator {
        private static final PlainFieldsUpdater PLAIN_FIELD_UPDATER = new PlainFieldsUpdater();

        @Override
        public void apply(IndexedCluster oldCluster, IndexedCluster newCluster, IndexFieldsSerializer instance,
                ListF<FieldChange> destination)
        {
            destination.add(FieldChange.put(instance.key, instance.toDataFieldMapperF.apply(newCluster)));
        }

        @Override
        public boolean isSimpleDeletion() {
            return true;
        }
    }

    private static class LocationsFieldUpdater implements FieldsUpdateCalculator {

        @Override
        public void apply(IndexedCluster oldCluster, IndexedCluster newCluster, IndexFieldsSerializer instance,
                ListF<FieldChange> destination)
        {
            if (!oldCluster.getPlaces().equals(newCluster.getPlaces())) {
                if (oldCluster.getPlaces().isEmpty()) {
                    DataField insertList = DataField.list(newCluster.getPlaces()
                            .map(IndexToDeltaMapper::localizedStringDictionaryToField));
                    destination.add(FieldChange.put(instance.key, insertList));
                } else if (newCluster.getPlaces().isEmpty()) {
                    for (int i = oldCluster.getPlaces().size() - 1; i >= 0; i--) {
                        destination.add(FieldChange.deleteListItem(instance.key, i));
                    }
                } else {
                    destination.addAll(updateListField(instance.key, oldCluster.getPlaces(), newCluster.getPlaces()));
                }
            }
        }

        private static  ListF<FieldChange> updateListField(String key, ListF<LocalizedStringDictionary> oldValue,
                ListF<LocalizedStringDictionary> newValue)
        {
            ListF<ListDiffCalculator.Diff<LocalizedStringDictionary>> diffs = ListDiffCalculator.calculateDifference(oldValue, newValue);
            ListF<FieldChange> fieldChanges = Cf.arrayList(diffs.size());
            for (ListDiffCalculator.Diff<LocalizedStringDictionary> diff: diffs) {
                int index = diff.getPosition();
                if (diff instanceof ListDiffCalculator.ReplaceDiff) {
                    ListDiffCalculator.ReplaceDiff<LocalizedStringDictionary> diffTs =
                            (ListDiffCalculator.ReplaceDiff<LocalizedStringDictionary>) diff;
                    fieldChanges.add(FieldChange.putListItem(key, index,
                            localizedStringDictionaryToField(diffTs.getValue())));
                } else if (diff instanceof ListDiffCalculator.InsertDiff) {
                    ListDiffCalculator.InsertDiff<LocalizedStringDictionary> diffTs =
                            (ListDiffCalculator.InsertDiff<LocalizedStringDictionary>) diff;
                    fieldChanges.add(FieldChange.insertListItem(key, index,
                            localizedStringDictionaryToField(diffTs.getValue())));
                } else if (diff instanceof ListDiffCalculator.RemoveDiff) {
                    fieldChanges.add(FieldChange.deleteListItem(key, index));
                }
            }
            return fieldChanges;
        }

        @Override
        public boolean isSimpleDeletion() {
            return false;
        }
    }
}
