package ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.support.RecordFieldUtils;
import ru.yandex.chemodan.app.smartcache.AlbumsUtils;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DisplayedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.ClusterDiff;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.MatchedClustersDiff;
import ru.yandex.chemodan.app.smartcache.worker.utils.SortedSplitter;

/**
 * @author osidorkin
 */
public class IndexToDeltaMapper {
    public static final String INDEX_COLLECTION_ID = "_index";

    public static ListF<RecordChange> generateClusterListDelta(ListF<ClusterDiff> clusterDiffs) {
        return clusterDiffs.flatMap(IndexToDeltaMapper::generateClusterDelta);
    }

    public static ListF<RecordChange> generateClusterDelta(ClusterDiff diff) {
        switch (diff.getType()) {
        case ADD:
            return insertClusterToSnapshot(diff.getCluster());
        case REMOVE:
            return removeClusterFromSnapshot(diff.getCluster());
        case MAJORITY_MATCH:
            return updateCluster((MatchedClustersDiff) diff);
        default:
            return Cf.list();
        }
    }

    public static ListF<RecordChange> insertClusterListToSnapshot(ListF<IndexedCluster> clusters) {
        return clusters.flatMap(IndexToDeltaMapper::insertClusterToSnapshot);
    }

    public static ListF<RecordChange> removeClusterListFromSnapshot(ListF<IndexedCluster> clusters) {
        return clusters.flatMap(IndexToDeltaMapper::removeClusterFromSnapshot);
    }

    public static ListF<RecordChange> insertClusterToSnapshot(IndexedCluster cluster) {
        return Cf.wrap(insertClusterToSnapshotStream(cluster).collect(Collectors.toList()));
    }

    public static ListF<RecordChange> updateCluster(MatchedClustersDiff diff) {
        IndexedCluster oldCluster = diff.getCluster();
        IndexedCluster newCluster = diff.getNewCluster();
        String clusterId = oldCluster.getIdForDb();

        PhotoRecordId.Format format = oldCluster.getId().getPhotoIdFormat();

        MapF<String, PhotoViewLuceneInfoPojo> clusterPhotos = oldCluster.getPhotos()
                .toMapMappingToKey(photo -> generatePhotoId(photo, format));
        MapF<String, PhotoViewLuceneInfoPojo> pojoPhotos = newCluster.getPhotos()
                .toMapMappingToKey(photo -> generatePhotoId(photo, format));

        SetF<String> addedKeys = pojoPhotos.keySet().minus(clusterPhotos.keySet());
        SetF<String> removedKeys = clusterPhotos.keySet().minus(pojoPhotos.keySet());
        SetF<String> matchedKeys = pojoPhotos.keySet().intersect(clusterPhotos.keySet());

        Stream<RecordChange> addedPhotos = addedKeys.stream().map(pojoPhotos::getTs)
                .map(photo -> insertPhoto(oldCluster, photo));
        Stream<RecordChange> removedPhotos = removedKeys.stream().map(key -> RecordChange.delete(clusterId, key));

        Stream<RecordChange> updatedPhotos = matchedKeys.stream()
            .map(id -> Tuple2.tuple(id, updatePhoto(clusterPhotos.getTs(id), pojoPhotos.getTs(id))))
            .filter(tuple -> tuple.get2().isNotEmpty())
            .map(changesTuple -> RecordChange.update(clusterId, changesTuple.get1(), changesTuple.get2()));

        Stream<RecordChange> allChanges = Stream.concat(addedPhotos, Stream.concat(removedPhotos, updatedPhotos));
        return Cf.toList(updateClusterIndex(oldCluster, newCluster))
                .plus(allChanges.collect(Collectors.toList()));

    }

    public static Option<RecordChange> updateClusterIndex(IndexedCluster oldCluster, IndexedCluster newCluster) {
        ListF<FieldChange> indexChanges = IndexFieldsSerializer.indexedClusterDiffToFields(oldCluster, newCluster);
        if (indexChanges.isEmpty()) {
            return Option.empty();
        }
        return Option.of(RecordChange.update(INDEX_COLLECTION_ID, oldCluster.getIdForDb(), indexChanges));
    }

    public static ListF<RecordChange> removeClusterFromSnapshot(IndexedCluster cluster) {
        return Cf.wrap(removeClusterFromSnapshotStream(cluster).collect(Collectors.toList()));
    }

    public static Stream<RecordChange> insertClusterToSnapshotStream(IndexedCluster cluster) {
        RecordChange indexChange = insertClusterToIndex(cluster);
        Stream<RecordChange> photos = cluster.getPhotos().stream().map(photo -> insertPhoto(cluster, photo));
        return Stream.concat(photos, Stream.of(indexChange));
    }

    public static Stream<RecordChange> removeClusterFromSnapshotStream(IndexedCluster cluster) {
        RecordChange indexChange = removeClusterFromIndex(cluster);
        Stream<RecordChange> photos = cluster.getPhotos().stream().map(photo -> removePhoto(cluster, photo));
        return Stream.concat(photos, Stream.of(indexChange));
    }

    private static RecordChange removeClusterFromIndex(IndexedCluster indexedCluster) {
        return RecordChange.delete(INDEX_COLLECTION_ID, indexedCluster.getIdForDb());
    }

    public static ListF<IndexedCluster> retrieveClusters(Snapshot snapshot) {
        return Cf.wrap(retrieveClustersStream(snapshot)
                .collect(Collectors.toList())
        );
    }

    private static Stream<IndexedCluster> retrieveClustersStream(Snapshot snapshot) {
        ListF<DataRecord> rows = snapshot.records().toList();
        if (rows.isEmpty()) {
            return Stream.empty();
        }
        MapF<String, ListF<DataRecord>> collections =
                SortedSplitter.sortedGroupBy(rows, DataRecord::getCollectionId);
        return collections.getTs(INDEX_COLLECTION_ID).stream()
                .map(row -> retrieveCluster(
                        row, collections.getOrElse(row.getRecordId(), Cf.list()), AlbumsUtils.isWithAlbums(snapshot)));
    }

    public static DisplayedCluster retrieveDisplayedCluster(DataRecord indexRow, boolean withAlbums) {
        return retrieveDisplayedCluster(indexRow.getRecordId(), indexRow.getData(), withAlbums);
    }

    public static DisplayedCluster retrieveDisplayedCluster(
            String clusterId, MapF<String, DataField> data, boolean withAlbums)
    {
        Instant from = data.getOrThrow(ClusterKeys.FROM_INSTANT).timestampValue();
        Instant to = data.getOrThrow(ClusterKeys.TO_INSTANT).timestampValue();
        int photosCount = data.getOrThrow(ClusterKeys.PHOTOS_COUNT).integerValue().intValue();
        Option<String> photosHash = data.getO(ClusterKeys.PHOTOS_HASH).map(DataField::stringValue);
        Option<LocalizedStringDictionary> cityO = data.getO(ClusterKeys.LOCALITY)
                .map(IndexToDeltaMapper::fieldToLocalizedStringDictionary);
        ListF<LocalizedStringDictionary> locations = data.getO(ClusterKeys.PLACES).map(DataField::listValue)
                                                            .getOrElse(Cf.list())
                                                            .map(IndexToDeltaMapper::fieldToLocalizedStringDictionary);

        MapF<AlbumType, Integer> albums;
        if (withAlbums) {
            albums = Cf.x(AlbumType.values()).zipWith(
                    type -> data.getO(IndexFieldsSerializer.getAlbumFieldName(type)).map(DataField::integerValue))
                    .map2(value -> value.getOrElse(0L).intValue()).filterBy2(value -> value > 0).toMap();
        } else {
            albums = Cf.map();
        }

        return new DisplayedCluster(
                ClusterId.parse(clusterId), from, to, photosCount, photosHash, cityO, locations, albums);
    }

    private static IndexedCluster retrieveCluster(DataRecord indexRow, ListF<DataRecord> photosRows, boolean withAlbums) {
        DisplayedCluster displayedCluster = retrieveDisplayedCluster(indexRow, withAlbums);
        ListF<PhotoViewLuceneInfoPojo> photos = retrievePhotos(photosRows, withAlbums);
        return IndexedCluster.consFromDataApi(displayedCluster, photos);
    }

    private static ListF<PhotoViewLuceneInfoPojo> retrievePhotos(ListF<DataRecord> photoListRows, boolean withAlbums) {
        return photoListRows.map(rec -> retrievePhoto(rec, withAlbums));
    }

    private static PhotoViewLuceneInfoPojo retrievePhoto(DataRecord photoRow, boolean withAlbums) {
        PhotoRecordId photoId = PhotoRecordId.parse(photoRow.getRecordId());

        return new PhotoViewLuceneInfoPojo(
                photoId.fileId, PhotoFields.PATH.get(photoRow), photoId.date,
                PhotoFields.LONGITUDE.getO(photoRow), PhotoFields.LATITUDE.getO(photoRow),
                PhotoFields.WIDTH.getO(photoRow), PhotoFields.HEIGHT.getO(photoRow),
                PhotoFields.BEAUTY.getO(photoRow), photoId.version,
                Option.empty(), Option.empty(), Option.empty(), !withAlbums,
                PhotoFields.ALBUMS.getO(photoRow).getOrElse(Cf.set()));
    }

    private static RecordChange insertClusterToIndex(IndexedCluster indexedCluster) {
        return RecordChange.insert(INDEX_COLLECTION_ID, indexedCluster.getIdForDb(),
                IndexFieldsSerializer.indexedClusterToFields(indexedCluster));
    }

    private static ListF<FieldChange> updatePhoto(PhotoViewLuceneInfoPojo oldPhoto, PhotoViewLuceneInfoPojo newPhoto) {
        MapF<String, DataField> newFields = collectFields(newPhoto);
        ListF<FieldChange> changes = RecordFieldUtils.diff(collectFields(oldPhoto), newFields);

        if (changes.isNotEmpty()) {
            return changes.filter(ch -> ch.type == FieldChange.FieldChangeType.DELETE)
                    .plus(newFields.mapEntries(FieldChange::put));
        } else {
            return Cf.list();
        }
    }

    public static MapF<String, DataField> collectFields(PhotoViewLuceneInfoPojo photo) {
        MapF<String, DataField> result = Cf.linkedHashMap();

        result.put(PhotoFields.PATH.toData(photo.key));
        result.putAll(photo.latitude.map(PhotoFields.LATITUDE::toData));
        result.putAll(photo.longitude.map(PhotoFields.LONGITUDE::toData));
        result.putAll(photo.width.map(PhotoFields.WIDTH::toData));
        result.putAll(photo.height.map(PhotoFields.HEIGHT::toData));
        result.putAll(photo.beauty.map(PhotoFields.BEAUTY::toData));

        SetF<AlbumType> albums = photo.getAlbums();
        result.putAll(Option.when(albums.isNotEmpty(), albums).map(PhotoFields.ALBUMS::toData));

        return result;
    }

    private static RecordChange removePhoto(IndexedCluster cluster, PhotoViewLuceneInfoPojo photo) {
        return RecordChange.delete(cluster.getIdForDb(), generatePhotoId(photo, cluster.getId().getPhotoIdFormat()));
    }

    private static RecordChange insertPhoto(IndexedCluster cluster, PhotoViewLuceneInfoPojo photo) {
        return RecordChange.insert(cluster.getIdForDb(), generatePhotoId(photo, cluster.getId().getPhotoIdFormat()),
                collectFields(photo));
    }

    private static String generatePhotoId(PhotoViewLuceneInfoPojo photo, PhotoRecordId.Format format) {
        return PhotoRecordId.of(photo).format(format);
    }

    static DataField localizedStringDictionaryToField(LocalizedStringDictionary dict) {
        return DataField.list(dict.toList().map(DataField::string));
    }

    public static LocalizedStringDictionary fieldToLocalizedStringDictionary(DataField field) {
        return new LocalizedStringDictionary(field.listValue().map(DataField::stringValue));
    }
}
