package ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecords;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange.FieldChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.ClusterDiff;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.MatchedClustersDiff;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
public class IndexToDeltaMapperTest {

    @Test
    public void testListChanges() {
        assertPlacesDiff(Cf.list("place0", "place1", "place2", "place3"), Cf.list("place1", "place4", "place6"));
    }

    @Test
    public void testListInsert() {
        assertPlacesDiff(Cf.list(), Cf.list("place1", "place4", "place6"));
    }

    private static void assertPlacesDiff(ListF<String> places1, ListF<String> places2) {
        ListF<LocalizedStringDictionary> oldPlaces = places1
                .map(TestUtils::createLocalizedStringDictionary);
        ListF<LocalizedStringDictionary> newPlaces = places2
                .map(TestUtils::createLocalizedStringDictionary);

        IndexedCluster oldCluster = IndexedCluster.consFromLuceneData(ClusterId.Format.STALE,
                instant(300), instant(400), 0,  Option.empty(), Cf.list())
                .withGeoLocality(Option.of(new LocalizedStringDictionary(Cf.list("Город", "City"))))
                .withGeoPlaces(oldPlaces);

        Delta initial = new Delta(IndexToDeltaMapper.insertClusterToSnapshot(oldCluster)).withRev(0);
        DataRecords initialRecords = applyDelta(DataRecords.EMPTY, initial);
        assertPlacesEquals(initialRecords, oldPlaces);

        IndexedCluster newCluster = oldCluster.withGeoPlaces(newPlaces);

        ClusterDiff diff = MatchedClustersDiff.majorityMatch(oldCluster, newCluster, 1);
        Delta update = new Delta(IndexToDeltaMapper.generateClusterDelta(diff)).withRev(1);
        assertNoPlacesDelete(update);
        DataRecords updatedRecords = applyDelta(initialRecords, update);
        assertPlacesEquals(updatedRecords, newPlaces);

        ClusterDiff diffReverse = MatchedClustersDiff.majorityMatch(newCluster, oldCluster, 1);
        Delta updateReverse = new Delta(IndexToDeltaMapper.generateClusterDelta(diffReverse)).withRev(2);
        assertNoPlacesDelete(updateReverse);
        DataRecords reversedRecords = applyDelta(updatedRecords, updateReverse);
        assertPlacesEquals(reversedRecords, oldPlaces);
    }

    private static void assertNoPlacesDelete(Delta delta) {
        Assert.isFalse(delta.changes.stream()
            .flatMap(recordChange -> recordChange.fieldChanges.stream())
            .filter(fieldChange -> ClusterKeys.PLACES.equals(fieldChange.key))
            .anyMatch(fieldChange -> fieldChange.type == FieldChangeType.DELETE));
    }

    private static DataRecords applyDelta(DataRecords records, Delta delta) {
        Database database = Database.consNew(
                DataApiUserId.parse("123"),
                DatabaseHandle.consGlobal("dbId", "handle"))
                .withIncRev();
        return new Snapshot(database, records)
                .toPatchable()
                .patch(delta)
                .unmodifiable()
                .records;
    }

    private static void assertPlacesEquals(DataRecords records, ListF<LocalizedStringDictionary> values) {
        DataRecord initialRecord = records.records().iterator().next();
        ListF<DataField> places = initialRecord.getData().getO(ClusterKeys.PLACES).map(DataField::listValue)
                .getOrElse(Cf.list());
        ListF<LocalizedStringDictionary> locations = places.map(IndexToDeltaMapper::fieldToLocalizedStringDictionary);
        Assert.equals(values, locations);
    }

    @Test
    public void testIndexChanges() {
        IndexedCluster oldCluster = IndexedCluster.consFromLuceneData(ClusterId.Format.STALE,
                instant(100), instant(200), 1, Option.empty(), Cf.list());

        IndexedCluster newCluster = IndexedCluster.consFromLuceneData(ClusterId.Format.STALE,
                instant(300), instant(400), 0, Option.empty(), Cf.list())
                .withGeoLocality(Option.of(new LocalizedStringDictionary(Cf.list("Город", "City"))));

        ClusterDiff diff = MatchedClustersDiff.majorityMatch(oldCluster, newCluster, 1);
        ListF<RecordChange> recordChanges = IndexToDeltaMapper.generateClusterDelta(diff);
        Assert.equals(1, recordChanges.size());
        RecordChange indexChange = recordChanges.first();
        Assert.equals(RecordChangeType.UPDATE, indexChange.type);
        Assert.equals(IndexToDeltaMapper.INDEX_COLLECTION_ID, indexChange.collectionId);
        MapF<String, FieldChange> fieldChanges = indexChange.fieldChanges.toMapMappingToKey(fieldChange -> fieldChange.key);
        Assert.equals(4, fieldChanges.size());
        assertFieldChange(fieldChanges, ClusterKeys.FROM_INSTANT, FieldChangeType.PUT, Option.of(DataField.timestamp(instant(300))));
        assertFieldChange(fieldChanges, ClusterKeys.TO_INSTANT, FieldChangeType.PUT, Option.of(DataField.timestamp(instant(400))));
        assertFieldChange(fieldChanges, ClusterKeys.PHOTOS_COUNT, FieldChangeType.PUT, Option.of(DataField.integer(0)));
        assertFieldChange(fieldChanges, ClusterKeys.LOCALITY, FieldChangeType.PUT, newCluster.getLocalityO()
                .map(IndexFieldsSerializer::localizedStringDictionaryToField));
    }

    @Test
    public void testIndexInsert() {
        IndexedCluster cluster = IndexedCluster.consFromLuceneData(ClusterId.Format.STALE,
                instant(300), instant(400), 0, Option.empty(), Cf.list())
                .withGeoLocality(Option.of(new LocalizedStringDictionary(Cf.list("Город", "City"))));
        ListF<RecordChange> recordChanges = IndexToDeltaMapper.insertClusterToSnapshot(cluster);
        Assert.equals(1, recordChanges.size());
        RecordChange indexChange = recordChanges.first();
        Assert.equals(RecordChangeType.INSERT, indexChange.type);
        Assert.equals(IndexToDeltaMapper.INDEX_COLLECTION_ID, indexChange.collectionId);
        MapF<String, FieldChange> fieldChanges = indexChange.fieldChanges.toMapMappingToKey(fieldChange -> fieldChange.key);
        Assert.equals(4, fieldChanges.size());
        assertFieldChange(fieldChanges, ClusterKeys.FROM_INSTANT, FieldChangeType.PUT, Option.of(DataField.timestamp(instant(300))));
        assertFieldChange(fieldChanges, ClusterKeys.TO_INSTANT, FieldChangeType.PUT, Option.of(DataField.timestamp(instant(400))));
        assertFieldChange(fieldChanges, ClusterKeys.PHOTOS_COUNT, FieldChangeType.PUT, Option.of(DataField.integer(0)));
        assertFieldChange(fieldChanges, ClusterKeys.LOCALITY, FieldChangeType.PUT, cluster.getLocalityO()
                .map(IndexFieldsSerializer::localizedStringDictionaryToField));

    }

    private static void assertFieldChange(MapF<String, FieldChange> fieldChanges, String key, FieldChangeType type,
            Option<DataField> valueO)
    {
        Assert.isTrue(fieldChanges.containsKeyTs(key));
        FieldChange change = fieldChanges.getTs(key);
        Assert.equals(type, change.type);
        if (valueO.isPresent()) {
            DataField value = change.getValue();
            Assert.equals(valueO.get(), value);
        }
    }

    private static Instant instant(long instant) {
        return new Instant(instant);
    }

}
