package ru.yandex.chemodan.app.smartcache.worker.processing;

import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Stream;

import javax.annotation.PostConstruct;

import yandex.maps.proto.common2.geo_object.GeoObjectOuterClass;
import yandex.maps.proto.common2.response.ResponseOuterClass;
import yandex.maps.proto.search.address.AddressOuterClass;
import yandex.maps.proto.search.kind.KindOuterClass;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.app.smartcache.worker.utils.TopNSelector;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.geosearch.GeosearchClient;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.concurrent.RpsLimiter;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author osidorkin
 */
public class GeocoderManager {

    private static final Comparator<LocalizedStringDictionary> DEFAULT_RESOURCE_COMPARATOR =
            Comparator.naturalComparator().compose(LocalizedStringDictionary::getDefaultResource);

    private static final Logger logger = LoggerFactory.getLogger(GeocoderManager.class);

    private AtomicReference<Tuple2List<Pattern, String>> shortenPatternsCompiled =
            new AtomicReference<>(Tuple2List.tuple2List());

    private final DynamicPropertyManager dynamicPropertyManager;
    private final RpsLimiter rpsLimiter;
    private final RpsLimiter geosearchRpsLimiter;

    private final GeosearchClient geosearchClient;

    private final MeteredCache<Coordinates, GeoPlaceLocalizedNames> cache;


    public GeocoderManager(
            GeosearchClient geosearchClient, DynamicPropertyManager dynamicPropertyManager,
            MeteredCache<Coordinates, GeoPlaceLocalizedNames> cache)
    {
        this.geosearchClient = geosearchClient;
        this.dynamicPropertyManager = dynamicPropertyManager;
        this.rpsLimiter = new RpsLimiter(DynamicVars.geocoderRPSMax.get());
        this.geosearchRpsLimiter = new RpsLimiter(DynamicVars.geocoderRPSMax.get());
        this.cache = cache;
    }

    @PostConstruct
    public void init() {
        dynamicPropertyManager.registerAndFireWatcher(DynamicVars.geocoderRPSMax, (value) -> {
            logger.info("RPS limit set to {}", value);
            rpsLimiter.setRpsLimit(value);
            geosearchRpsLimiter.setRpsLimit(value);
        });

        dynamicPropertyManager.registerAndFireWatcher(DynamicVars.shortenNamesPatterns, (value) -> {
            logger.info("Shorten names patterns set to {}", value);
            compilePatterns(value);
        });
    }

    public static class LocalCache {
        private final MapF<Coordinates, GeoPlaceLocalizedNames> places = Cf.hashMap();
    }

    public IndexedCluster addGeocoderDataCached(IndexedCluster cluster, LocalCache cache) {
        ListF<GeoPlaceLocalizedNames> names = cluster.getPhotos().iterator()
                .filterMap(PhotoViewLuceneInfoPojo::getCoordinatesO)
                .map(coordinates -> cache.places.computeIfAbsent(coordinates, this::getGeoNames))
                .toList();

        Option<LocalizedStringDictionary> locality = getTopClusterLocality(
                names.iterator().filterMap(GeoPlaceLocalizedNames::getLocality).stream());

        ListF<LocalizedStringDictionary> places = getTopClusterStreets(
                names.iterator().filterMap(GeoPlaceLocalizedNames::getStreet).stream());

        return cluster
                .withGeoLocality(locality.map(this::shortenSinglePlaceName))
                .withGeoPlaces(places.map(this::shortenSinglePlaceName));
    }

    ListF<LocalizedStringDictionary> shortenPlaceNames(ListF<LocalizedStringDictionary> input) {
        return input.map(this::shortenSinglePlaceName);
    }

    private LocalizedStringDictionary shortenSinglePlaceName(LocalizedStringDictionary input) {
        return new LocalizedStringDictionary(input
                .toMap()
                .mapValuesWithKey((language, s) -> language == Language.RUSSIAN ? shortenPlaceName(s) : s)
        );
    }

    String shortenPlaceName(String name) {
        for (Tuple2<Pattern, String> entry : shortenPatternsCompiled.get()) {
            name = entry._1.matcher(name).replaceAll(entry._2);
        }
        return name;
    }

    private Option<LocalizedStringDictionary> getTopClusterLocality(Stream<LocalizedStringDictionary> values) {
        return TopNSelector.getTopNItems(values, DEFAULT_RESOURCE_COMPARATOR, 1).singleO();
    }

    private ListF<LocalizedStringDictionary> getTopClusterStreets(Stream<LocalizedStringDictionary> values) {
        return TopNSelector.getTopNItems(values, DEFAULT_RESOURCE_COMPARATOR, DynamicVars.photoslicePlacesNum.get());
    }

    private GeoPlaceLocalizedNames getGeoNames(Coordinates coordinates) {
        if (coordinates.getLatitude() <= -90 || coordinates.getLatitude() >= 90) {
            return GeoPlaceLocalizedNames.EMPTY;
        }
        Option<GeoPlaceLocalizedNames> fromCache = cache.getO(coordinates);

        if (fromCache.isPresent()) {
            cache.incHits();
            return fromCache.get();
        }

        GeoPlaceLocalizedNames street = getGeoNames(GeoType.STREET, coordinates);
        GeoPlaceLocalizedNames result = street.isEmpty() ? getGeoNames(GeoType.LOCALITY, coordinates) : street;

        cache.put(coordinates, result);
        return result;
    }

    private GeoPlaceLocalizedNames getGeoNames(GeoType type, Coordinates coordinates) {
        ResponseOuterClass.Response response = geosearchRpsLimiter.execute(() -> geosearchClient.geocode(
                coordinates, LocalizedStringDictionary.languages.map(Language::value),
                Option.of(1), Option.empty(), Option.when(type == GeoType.STREET, KindOuterClass.Kind.STREET)));

        ListF<GeoObjectOuterClass.GeoObject> objects = Cf.x(response.getReply().getGeoObjectList());

        if (objects.isEmpty()) {
            return GeoPlaceLocalizedNames.EMPTY;
        } else if (type == GeoType.STREET) {
            return new GeoPlaceLocalizedNames(GeoType.LOCALITY.extractNames(objects), GeoType.STREET.extractNames(objects));
        } else {
            return new GeoPlaceLocalizedNames(GeoType.LOCALITY.extractNames(objects), Option.empty());
        }
    }

    void compilePatterns(ListF<String> newValue) {
        if (newValue.length() % 2 != 0) {
            logger.warn("Bad shorten patterns. Length must be divisible by 2. Current value: {}", newValue);
            return;
        }
        Tuple2List<Pattern, String> compiledPatterns = Tuple2List.arrayList();
        for (int i = 0; i < newValue.length(); i += 2) {
            String pattern = newValue.get(i);
            String replacement = newValue.get(i + 1);
            try {
                Pattern compiled = Pattern.compile(pattern);
                compiledPatterns.add(Tuple2.tuple(compiled, replacement));
            } catch (PatternSyntaxException e) {
                logger.warn("Cannot compile pattern {}", pattern);
                return;
            }
        }
        shortenPatternsCompiled.set(compiledPatterns);
    }

    private enum GeoType {
        STREET(geoObject -> Option.of(geoObject.getName())),
        LOCALITY(geoObject -> Option.wrap(GeosearchClient.extractGeoObjectMetadata(geoObject)).flatMapO(meta -> {
            ListF<AddressOuterClass.Component> components = Cf.x(meta.getAddress().getComponentList());

            return components.find(c -> Cf.x(c.getKindList()).containsTs(KindOuterClass.Kind.LOCALITY))
                    .orElse(components.filter(c -> Cf.x(c.getKindList()).containsTs(KindOuterClass.Kind.AREA)).lastO())
                    .orElse(components.filter(c -> Cf.x(c.getKindList()).containsTs(KindOuterClass.Kind.PROVINCE)).lastO())
                    .orElse(components.find(c -> Cf.x(c.getKindList()).containsTs(KindOuterClass.Kind.COUNTRY)))
                    .map(AddressOuterClass.Component::getName);
        }));

        private final Function<GeoObjectOuterClass.GeoObject, Option<String>> nameExtractor;

        GeoType(Function<GeoObjectOuterClass.GeoObject, Option<String>> nameExtractor) {
            this.nameExtractor = nameExtractor;
        }

        public Option<LocalizedStringDictionary> extractNames(ListF<GeoObjectOuterClass.GeoObject> geoObjects) {
            ListF<String> extracted = geoObjects.filterMap(nameExtractor);

            return Option.when(extracted.isNotEmpty(), () -> new LocalizedStringDictionary(extracted));
        }
    }
}
