package ru.yandex.chemodan.app.smartcache.worker.processing;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import yandex.maps.proto.common2.response.ResponseOuterClass;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.geosearch.GeosearchClient;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class GeocoderManagerTest {

    @Mock
    private GeosearchClient geosearchClient;

    private GeocoderManager manager;

    @Before
    public void setup() {
        manager = new GeocoderManager(geosearchClient, Mockito.mock(DynamicPropertyManager.class), MeteredCache.noCache());

        ListF<String> shortenNamesPatterns = Cf.list(
                "(.*\\b[Уу]л)ица\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Пп]росп)ект\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Пп]ер)еулок\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Пп]л)ощадь\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Шш])оссе\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Аа]л)лея\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Бб]ульв)ар\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Рр])ека\\b\\.?(.*)", "$1.$2",
                "(.*\\b[Пп])роезд\\b(.*)", "$1р-д$2"
        );
        manager.compilePatterns(shortenNamesPatterns);

    }

    @Test
    public void shortenPlaceNamesTest() {
        ListF<LocalizedStringDictionary> input = Cf.list(
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.ENGLISH, "Novaya street",
                        Language.RUSSIAN, "Новая улица"
                )),
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.TURKISH, "asdf",
                        Language.RUSSIAN, "Центральная площадь"
                )),
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.UKRAINIAN, "New вулица"
                ))
        );
        ListF<LocalizedStringDictionary> output = manager.shortenPlaceNames(input);
        ListF<LocalizedStringDictionary> expectedOutput = Cf.list(
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.ENGLISH, "Novaya street",
                        Language.RUSSIAN, "Новая ул."
                )),
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.TURKISH, "asdf",
                        Language.RUSSIAN, "Центральная пл."
                )),
                new LocalizedStringDictionary(Tuple2List.fromPairs(
                        Language.UKRAINIAN, "New вулица"
                ))
        );

        Assert.assertListsEqual(expectedOutput, output);

    }

    @Test
    public void shortenTest() {
        Assert.equals("", manager.shortenPlaceName(""));
        Assert.equals("Москва, Садовническая набережная", manager.shortenPlaceName("Москва, Садовническая набережная"));

        Assert.equals("ул.", manager.shortenPlaceName("улица"));
        Assert.equals("Ул.", manager.shortenPlaceName("Улица"));
        Assert.equals(" ул.", manager.shortenPlaceName(" улица"));
        Assert.equals("Новая Ул.", manager.shortenPlaceName("Новая Улица"));
        Assert.equals("Новая лица", manager.shortenPlaceName("Новая лица"));
        Assert.equals("Старая ул.", manager.shortenPlaceName("Старая улица."));
        Assert.equals("Старая ул., д.1", manager.shortenPlaceName("Старая улица, д.1"));
        Assert.equals("Старая алица, д.1", manager.shortenPlaceName("Старая алица, д.1"));
        Assert.equals("улицах", manager.shortenPlaceName("улицах"));
        Assert.equals(", улицах", manager.shortenPlaceName(", улицах"));
        Assert.equals("улицах.", manager.shortenPlaceName("улицах."));
        Assert.equals(", улицах, ", manager.shortenPlaceName(", улицах, "));

        Assert.equals("Старый просп.", manager.shortenPlaceName("Старый проспект."));
        Assert.equals("Старый пер.", manager.shortenPlaceName("Старый переулок."));
        Assert.equals("Старая пл.", manager.shortenPlaceName("Старая площадь."));
        Assert.equals("Старое ш.", manager.shortenPlaceName("Старое шоссе."));
        Assert.equals("Старая ал.", manager.shortenPlaceName("Старая аллея."));
        Assert.equals("Старый бульв.", manager.shortenPlaceName("Старый бульвар."));
        Assert.equals("Старая р.", manager.shortenPlaceName("Старая река."));

        Assert.equals("пр-д", manager.shortenPlaceName("проезд"));
        Assert.equals("Пр-д", manager.shortenPlaceName("Проезд"));
        Assert.equals(" пр-д", manager.shortenPlaceName(" проезд"));
        Assert.equals("Новый Пр-д", manager.shortenPlaceName("Новый Проезд"));
        Assert.equals("Новый приезд", manager.shortenPlaceName("Новый приезд"));
        Assert.equals("Старый пр-д.", manager.shortenPlaceName("Старый проезд."));
        Assert.equals("Старый пр-д, д.1", manager.shortenPlaceName("Старый проезд, д.1"));
        Assert.equals("Старый приезд, д.1", manager.shortenPlaceName("Старый приезд, д.1"));
        Assert.equals("проездом", manager.shortenPlaceName("проездом"));
        Assert.equals(", проезды", manager.shortenPlaceName(", проезды"));
        Assert.equals("проездо.", manager.shortenPlaceName("проездо."));
        Assert.equals(", проездах, ", manager.shortenPlaceName(", проездах, "));

    }

    @Test
    public void simple() {
        PhotoViewLuceneInfoPojo photo1 = new PhotoViewLuceneInfoPojo(
                "", "", Instant.now(), Option.of(1.), Option.of(2.), 1);

        PhotoViewLuceneInfoPojo photo2 = new PhotoViewLuceneInfoPojo(
                "", "", Instant.now(), Option.of(3.), Option.of(4.), 1);

        IndexedCluster cluster = IndexedCluster.consFromLuceneData(ClusterId.Format.STALE,
                Instant.now(), Instant.now(), 1, Option.empty(), Cf.list(photo2, photo1));

        ArgumentCaptor<Coordinates> coordinatesCaptor = ArgumentCaptor.forClass(Coordinates.class);
        ResponseOuterClass.Response collection = emptyGeoCollection();

        Mockito.when(geosearchClient.geocode(coordinatesCaptor.capture(),
                anyList(), anyOption(), anyOption(), anyOption())).thenReturn(collection);

        manager.addGeocoderDataCached(cluster, new GeocoderManager.LocalCache());

        Assert.equals(Cf.list(
                new Coordinates(4, 3), new Coordinates(4, 3),
                new Coordinates(2, 1), new Coordinates(2, 1)),
                coordinatesCaptor.getAllValues());
    }

    private static ResponseOuterClass.Response emptyGeoCollection() {
        return ResponseOuterClass.Response
                .newBuilder()
                .build();
    }

    private static <A> Option<A> anyOption() {
        return Mockito.any(optionClass());
    }

    private static <A> Class<Option<A>> optionClass() {
        return ClassX.wrap(Option.class).<Option<A>>uncheckedCast().getClazz();
    }

    private static <A> ListF<A> anyList() {
        return Mockito.any();
    }
}
