package ru.yandex.chemodan.app.smartcache.worker.processing;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DisplayedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestsContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
@ContextConfiguration(classes = {
        TestsContextConfiguration.class
})
@TestExecutionListeners(value = {
        DependencyInjectionTestExecutionListener.class
})
@ActivateDataApiEmbeddedPg
public class PhotoSliceProcessingManagerTest extends DataApiTestSupport {


    @Autowired
    private PhotosliceProcessingManager photosliceProcessingManager;
    @Autowired
    private DataApiStorageManager dataApiStorageManager;
    @Autowired
    private MpfsClient mpfsClient;

    private DataApiUserId testUid;

    @Before
    public void before() {
        testUid = createRandomCleanUserInDefaultShard();
        Mockito.when(mpfsClient.isQuickMoveEnabled(Mockito.anyString())).thenReturn(false);
    }

    @Test
    public void testPhotoRename() {
        PhotoViewLuceneClusterPojo initialCluster = createCluster(getCommonPhotos().plus(getLastPhoto("name1")).reverse());
        photosliceProcessingManager.storeSnapshot(testUid, Cf.list(initialCluster), false);

        PhotoViewLuceneClusterPojo updatedCluster = createCluster(getCommonPhotos().plus(getLastPhoto("name2")).reverse());
        photosliceProcessingManager.updateSnapshot(testUid, Cf.list(updatedCluster));

        Assert.equals("name2", loadClusters().single().getPhotos().last().key);
    }

    @Test
    public void skipExactlyMatchedCluster() {
        PhotoViewLuceneClusterPojo cluster = createCluster(getCommonPhotos().reverse());

        PhotoViewLuceneClusterPojo skipped = new PhotoViewLuceneClusterPojo(0, cluster.min, cluster.max, Cf.list())
                .setPhotosHashForTestReturnThis(cluster.getPhotosHash());

        dataApiStorageManager.createDatabaseWithAlbums(testUid);

        photosliceProcessingManager.updateSnapshot(testUid, Cf.list(createCluster(Cf.list(getLastPhoto("X"))), cluster));
        Assert.hasSize(2, loadClusters());

        photosliceProcessingManager.updateSnapshot(testUid, Cf.list(skipped));
        Assert.equals(cluster.size, loadClusters().single().getPhotosCount());
    }

    @Test
    public void loadUnloadedCluster() {
        Function<Integer, PhotoViewLuceneInfoPojo> createPhoto = ts ->
                new PhotoViewLuceneInfoPojo("" + ts, "" + ts, instant(ts), Option.empty(), Option.empty(), 0L);

        ListF<PhotoViewLuceneInfoPojo> initialPhotos = Cf.list(1000, 2000, 3000, 4000, 5000).map(createPhoto);

        ListF<PhotoViewLuceneInfoPojo> overlapPhotos = Cf.list(2000).map(createPhoto);
        ListF<PhotoViewLuceneInfoPojo> exactMatchingPhotos = Cf.list(3000, 4000, 5000).map(createPhoto);

        photosliceProcessingManager.storeSnapshot(testUid, Cf.list(createCluster(initialPhotos)), false);
        IndexedCluster created = loadClusters().single();

        photosliceProcessingManager.updateSnapshot(testUid, Cf.list(createCluster(exactMatchingPhotos)));
        Assert.equals(created.getId(), loadClusters().single().getId());

        PhotoViewLuceneClusterPojo cluster = createCluster(exactMatchingPhotos);

        PhotoViewLuceneClusterPojo matched = new PhotoViewLuceneClusterPojo(22, cluster.min, cluster.max, Cf.list())
                .setPhotosHashForTestReturnThis(cluster.getPhotosHash());

        photosliceProcessingManager.updateSnapshot(testUid, Cf.list(matched, createCluster(overlapPhotos)));

        Assert.isEmpty(dataApiStorageManager.getSnapshot(testUid, CollectionIdCondition.eq(created.getIdForDb())).records());
        Assert.equals(Cf.list(matched.size, overlapPhotos.size()), loadClusters().map(DisplayedCluster::getPhotosCount));
    }

    @Test
    public void reformatClusterIds() {
        Function<Integer, PhotoViewLuceneInfoPojo> createPhoto = ts ->
                new PhotoViewLuceneInfoPojo("" + ts, "" + ts, instant(ts), Option.empty(), Option.empty(), 0L);

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                createCluster(Cf.list(3000, 4000).map(createPhoto)),
                createCluster(Cf.list(1000, 2000).map(createPhoto)),
                createCluster(Cf.list(-2000, -1000).map(createPhoto)));

        photosliceProcessingManager.storeSnapshot(testUid, clusters.firstO(), false);
        Assert.equals(Cf.set(ClusterId.Format.STALE), loadClusters().map(c -> c.getId().getFormat()).unique());

        photosliceProcessingManager.updateSnapshot(testUid, clusters.take(2));
        Assert.equals(Cf.set(ClusterId.Format.STALE), loadClusters().map(c -> c.getId().getFormat()).unique());

        photosliceProcessingManager.updateSnapshot(testUid, clusters);
        Assert.equals(Cf.set(ClusterId.Format.ADVANCED), loadClusters().map(c -> c.getId().getFormat()).unique());

        photosliceProcessingManager.dropSnapshot(testUid);

        photosliceProcessingManager.storeSnapshot(testUid, clusters.lastO(), false);
        Assert.equals(Cf.set(ClusterId.Format.ADVANCED), loadClusters().map(c -> c.getId().getFormat()).unique());

        photosliceProcessingManager.updateSnapshot(testUid, clusters.take(2));
        Assert.equals(Cf.set(ClusterId.Format.ADVANCED), loadClusters().map(c -> c.getId().getFormat()).unique());
    }

    private ListF<IndexedCluster> loadClusters() {
        return IndexToDeltaMapper.retrieveClusters(
                dataApiStorageManager.getSnapshot(testUid, CollectionIdCondition.all()));
    }

    private static PhotoViewLuceneClusterPojo createCluster(ListF<PhotoViewLuceneInfoPojo> photos) {
        return new PhotoViewLuceneClusterPojo(photos.size(), photos.first().date, photos.last().date, photos);
    }

    private static ListF<PhotoViewLuceneInfoPojo> getCommonPhotos() {
        return Cf.list(
                new PhotoViewLuceneInfoPojo("f236590b7ceaab57541d4e707d25ca91ab6135c150e3bf7c54fd161a45487602",
                        "/disk/Фотокамера/2012-06-06 23-20-58.JPG",
                        instant(1338996058000L), Option.empty(), Option.empty(), 1L),
                new PhotoViewLuceneInfoPojo("b4a173f8e5a084df8294ccb68b3569f5fece40c419ed35df5f1e2ece723b15b0",
                        "/disk/Фотокамера/2012-06-06 23-20-54.JPG",
                        instant(1338996054000L), Option.empty(), Option.empty(), 2L)
               );
    }

    private static PhotoViewLuceneInfoPojo getLastPhoto(String key) {
        return new PhotoViewLuceneInfoPojo("b25eef7508e634d14ebae1570481adb5968ba070c72e19ad4288f36c1c593e59",
                key, instant(1326833018000L), Option.empty(), Option.empty(), 3L);

    }

    private static Instant instant(long value) {
        return new Instant(value);
    }
}
