package ru.yandex.chemodan.app.smartcache.worker.processing;

import java.util.IntSummaryStatistics;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.dao.UncategorizedDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.smartcache.AlbumsUtils;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizerManager;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneResponsePojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.AlbumMetaPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DisplayedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.ClusterKeys;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.ClusterDiff;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.PhotoViewClusterMatcher;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.DiffType;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.MatchedClustersDiff;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.BlockerTaskParameters;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdateClusterPlacesTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceBlockerTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceSubmitterTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdatePhotosliceTask;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.UpdaterTaskParameters;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.app.smartcache.worker.utils.PhotoViewCacheBender;
import ru.yandex.chemodan.app.smartcache.worker.utils.PojoConverters;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author osidorkin
 */
public class PhotosliceProcessingManager {

    private static final Logger logger = LoggerFactory.getLogger(PhotosliceProcessingManager.class);
    private static final Duration SCHEDULING_DELAY = Duration.millis(10000);

    private final DataApiStorageManager dataApiStorageManager;
    private final ClusterizerManager clusterizerManager;
    private final GeocoderManager geocoderManager;
    private final CleanupManager cleanupManager;
    private final PgBazingaStorage bazingaStorage;
    private final MpfsClient mpfsClient;
    private final PhotosliceTransitionManager photosliceTransitionManager;

    private final ExecutorService submitPlacesUpdateExecuter;

    private static final CollectionIdCondition META_CONDITION = CollectionIdCondition.inSet(Cf.set(
            IndexToDeltaMapper.INDEX_COLLECTION_ID,
            AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID));

    public PhotosliceProcessingManager(DataApiStorageManager dataApiStorageManager,
            ClusterizerManager clusterizerManager, GeocoderManager geocoderManager,
            CleanupManager cleanupManager, PgBazingaStorage bazingaStorage,
            MpfsClient mpfsClient, PhotosliceTransitionManager photosliceTransitionManager)
    {
        this.dataApiStorageManager = dataApiStorageManager;
        this.clusterizerManager = clusterizerManager;
        this.geocoderManager = geocoderManager;
        this.cleanupManager = cleanupManager;
        this.bazingaStorage = bazingaStorage;

        this.mpfsClient = mpfsClient;
        this.photosliceTransitionManager = photosliceTransitionManager;

        //TODO: configurable
        this.submitPlacesUpdateExecuter = Executors.newFixedThreadPool(50);
    }

    void updateSnapshot(DataApiUserId uid, ListF<PhotoViewLuceneClusterPojo> clusterPojos) {
        updateSnapshot(uid, dataApiStorageManager.getSnapshot(uid, META_CONDITION), clusterPojos);
    }

    private ListF<IndexedCluster> loadHashMatchingClusters(
            Snapshot metaSnapshot, ListF<PhotoViewLuceneClusterPojo> clusterPojos)
    {
        SetF<String> luceneHashes = Cf.hashSetWithExpectedSize(clusterPojos.size());

        SetF<String> excludedCollectionIds = Cf.hashSet(
                IndexToDeltaMapper.INDEX_COLLECTION_ID, AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID);
        SetF<String> changedCollectionIds = Cf.hashSet();

        clusterPojos.forEach(p -> luceneHashes.add(p.getPhotosHash()));

        getIndexRecords(metaSnapshot).forEach(r -> {
            Option<String> hash = r.getData().getO(ClusterKeys.PHOTOS_HASH).map(DataField::stringValue);

            if (!hash.exists(luceneHashes::containsTs)) {
                changedCollectionIds.add(r.getRecordId());
            } else {
                excludedCollectionIds.add(r.getRecordId());
            }
        });

        Snapshot dataSnapshot = dataApiStorageManager.getSnapshot(
                metaSnapshot.database.uid, changedCollectionIds.size() <= excludedCollectionIds.size()
                        ? CollectionIdCondition.inSet(changedCollectionIds)
                        : CollectionIdCondition.notInSet(excludedCollectionIds));

        if (!dataSnapshot.database.handleValue().equals(metaSnapshot.database.handleValue())
                || dataSnapshot.database.rev > metaSnapshot.database.rev)
        {
            throw new OutdatedChangeException("Records snapshot is fresher than index snapshot");
        }
        dataSnapshot = dataSnapshot.withRecords(metaSnapshot.records().toList().plus(dataSnapshot.records()));

        return IndexToDeltaMapper.retrieveClusters(dataSnapshot);
    }

    private ListF<ClusterDiff> loadNotLoadedClusters(Snapshot metaSnapshot, ListF<ClusterDiff> diffs) {
        SetF<String> ids = diffs.filterMap(d -> Option.when(d.getCluster().isNotLoaded(), d.getCluster().getIdForDb())).unique();

        if (ids.isEmpty()) return diffs;

        Snapshot dataSnapshot = dataApiStorageManager.getSnapshot(
                metaSnapshot.database.uid, CollectionIdCondition.inSet(ids));

        if (!dataSnapshot.database.handleValue().equals(metaSnapshot.database.handleValue())
                || dataSnapshot.database.rev > metaSnapshot.database.rev)
        {
            throw new OutdatedChangeException("Records snapshot is fresher than index snapshot");
        }
        ListF<DataRecord> indexRecords = getIndexRecords(metaSnapshot).filter(r -> ids.containsTs(r.getRecordId())).toList();

        MapF<String, IndexedCluster> loadedById = IndexToDeltaMapper
                .retrieveClusters(dataSnapshot.withRecords(indexRecords.plus(dataSnapshot.records())))
                .toMapMappingToKey(DisplayedCluster::getIdForDb);

        return diffs.map(d -> d.withCluster(loadedById.getO(d.getCluster().getIdForDb()).getOrElse(d.getCluster())));
    }

    private void updateSnapshot(
            DataApiUserId uid, Snapshot metaSnapshot, ListF<PhotoViewLuceneClusterPojo> clusterPojos)
    {
        if (!AlbumsUtils.isWithAlbums(metaSnapshot)) {
            clusterPojos = clusterPojos.map(PhotoViewLuceneClusterPojo::withNoAlbums);
        }
        ListF<ClusterDiff> clusterDiffs = PhotoViewClusterMatcher
                .performMatching(loadHashMatchingClusters(metaSnapshot, clusterPojos), clusterPojos);

        if (clusterDiffs.isEmpty()) {
            return;
        }
        clusterDiffs = loadNotLoadedClusters(metaSnapshot, clusterDiffs);

        ListF<AlbumMetaPojo> albums = getAlbumRecords(metaSnapshot).map(AlbumsToDeltaMapper::parseDataRecord).toList();

        ListF<RecordChange> diffs = clusterDiffs.flatMap(IndexToDeltaMapper::generateClusterDelta)
                .plus(AlbumsToDeltaMapper.getAlbumsChanges(albums, clusterPojos));
        dataApiStorageManager.storeDelta(uid, metaSnapshot.database.rev, RevisionCheckMode.WHOLE_DATABASE, diffs);

        // Temporary hack for after-albums-init update
        SetF<String> reallyChangedClusters = diffs.map(change -> change.collectionId).unique();

        ListF<String> geoClusterIds = clusterDiffs.stream()
                .map(this::getClusterForPlacesUpdate)
                .filter(Option::isPresent).map(Option::get)
                .map(IndexedCluster::getIdForDb)
                .filter(reallyChangedClusters::containsTs)
                .collect(Collectors.toCollection(Cf::arrayList));

        if (geoClusterIds.isEmpty()) {
            return;
        }
        Option<Database> databaseO = dataApiStorageManager.getDatabaseO(uid);
        if (!databaseO.isPresent()) {
            return;
        }
        schedulePlacesUpdateAsync(databaseO.get(), geoClusterIds);
    }

    private void schedulePlacesUpdateAsync(Database database, ListF<String> geoClusterIds) {
        submitPlacesUpdateExecuter.submit(() -> {
            schedulePlacesUpdate(database, geoClusterIds);
        });
    }

    private Option<IndexedCluster> getClusterForPlacesUpdate(ClusterDiff diff) {
        if (diff.getType() == DiffType.ADD) {
            IndexedCluster cluster = diff.getCluster();
            return Option.when(cluster.hasPhotoWithCoordinates(), cluster);
        }
        if (diff.getType() == DiffType.MAJORITY_MATCH) {
            MatchedClustersDiff matchedDiff = (MatchedClustersDiff) diff;
            IndexedCluster cluster = matchedDiff.getCluster();
            // Use old cluster here since new cluster has different ID
            return Option.when(cluster.hasPhotoWithCoordinates() || matchedDiff.getNewCluster().hasPhotoWithCoordinates(), cluster);
        }
        return Option.empty();
    }

    //For scripting only
    public Option<ListF<IndexedCluster>> loadSnapshotO(DataApiUserId uid) {
        return dataApiStorageManager.getSnapshotO(uid, CollectionIdCondition.all()).map(IndexToDeltaMapper::retrieveClusters);
    }

    void storeSnapshot(DataApiUserId uid, ListF<PhotoViewLuceneClusterPojo> clusterPojos, boolean albumsEnabled) {
        ClusterId.Format format = ClusterId.Format.forLucene(clusterPojos);

        Database indexDatabase = albumsEnabled
                ? dataApiStorageManager.createDatabaseWithAlbums(uid)
                : dataApiStorageManager.createDatabase(uid);
        cleanupManager.registerCreatedDatabase(indexDatabase);

        if (!AlbumsUtils.isWithAlbums(indexDatabase)) {
            clusterPojos = clusterPojos.map(PhotoViewLuceneClusterPojo::withNoAlbums);
        }
        ListF<String> geoClusterIds = Cf.arrayList();
        ListF<RecordChange> dataApiSnapshot = clusterPojos.stream()
                .map(c -> PojoConverters.buildClustersIndex(format, c))
                .peek(cluster -> {
                    if (cluster.hasPhotoWithCoordinates()) {
                        geoClusterIds.add(cluster.getIdForDb());
                    }
                }).flatMap(IndexToDeltaMapper::insertClusterToSnapshotStream)
                .collect(Collectors.toCollection(Cf::arrayList))
                .plus(AlbumsToDeltaMapper.getAlbumsChanges(Cf.list(), clusterPojos));

        dataApiStorageManager.storeSnapshotIndex(indexDatabase, RevisionCheckMode.WHOLE_DATABASE, dataApiSnapshot);
        schedulePlacesUpdateAsync(indexDatabase, geoClusterIds);
    }

    public Either<Database, FullJobId> handleSnapshotInit(
            DataApiUserId uid, boolean async, boolean online, boolean albumsEnabled)
    {
        Option<Database> databaseO = dataApiStorageManager.getAndInitCopyOnWrite(uid);

        if (databaseO.exists(db -> !dataApiStorageManager.isAlbumsEnough(db, albumsEnabled))) {
            photosliceTransitionManager.initAlbums(uid, this);
            return Either.left(dataApiStorageManager.getAndInitCopyOnWrite(uid).get());
        } else if (databaseO.isPresent()) {
            return Either.left(databaseO.get());
        }
        if (async) {
            return Either.right(scheduleSnapshotUpdate(uid, Option.empty()));
        }
        updateSnapshotWithRetries(uid, online, albumsEnabled);

        return Either.left(dataApiStorageManager.getAndInitCopyOnWrite(uid).get());
    }

    public void handleSnapshotUpdate(DataApiUserId uid, Option<Instant> timestamp) {
        Option<Database> databaseO = dataApiStorageManager.getDatabaseO(uid);
        if (databaseO.isPresent()) {
            scheduleSnapshotUpdate(uid, timestamp);
        }
    }

    public FullJobId scheduleSnapshotUpdate(DataApiUserId uid, Option<Instant> timestamp) {
        Duration delay = DynamicVars.photosliceUpdateDelay.get();
        UpdaterTaskParameters updaterParams = new UpdaterTaskParameters(uid, timestamp, Option.of(Instant.now()));

        UpdatePhotosliceBlockerTask blockerTask = new UpdatePhotosliceBlockerTask(uid, delay);
        UpdatePhotosliceSubmitterTask submitterTask = new UpdatePhotosliceSubmitterTask(updaterParams);

        String blockerActiveUid = PgOnetimeUtils.getActiveUniqueIdentifier(blockerTask);
        Option<OnetimeJob> activeBlocker = bazingaStorage.findOnetimeJobByActiveUid(blockerActiveUid);

        Option<Instant> activeBlockUntil = activeBlocker.map(job -> {
            BlockerTaskParameters params = PgOnetimeUtils.parseParameters(blockerTask, job.getParameters());
            return job.getScheduleTime().minus(ObjectUtils.max(Duration.ZERO, params.delay.minus(delay)));
        });

        FullJobId addedId;

        if (!activeBlockUntil.exists(Instant.now()::isBefore)) {
            OnetimeJob updater = PgOnetimeUtils.makeJob(new UpdatePhotosliceTask(updaterParams), Instant.now());
            addedId = bazingaStorage.addOnetimeJob(updater, ActiveUidDuplicateBehavior.DO_NOTHING);

            if (addedId.getJobId().equals(updater.getJobId())) {
                OnetimeJob blocker = PgOnetimeUtils.makeJob(blockerTask, Instant.now().plus(delay));
                bazingaStorage.addOnetimeJob(blocker, ActiveUidDuplicateBehavior.MERGE);
            } else {
                OnetimeJob submitter = PgOnetimeUtils.makeJob(submitterTask, Instant.now().plus(delay));
                bazingaStorage.addOnetimeJob(submitter, ActiveUidDuplicateBehavior.DO_NOTHING);
            }
        } else {
            String submitterActiveUid = PgOnetimeUtils.getActiveUniqueIdentifier(submitterTask);
            Option<OnetimeJob> activeSubmitter = bazingaStorage.findOnetimeJobByActiveUid(submitterActiveUid);

            if (!activeSubmitter.exists(job -> job.getScheduleTime().isEqual(activeBlockUntil.get()))) {
                if (activeSubmitter.isPresent()) {
                    submitterTask = submitterTask.withParsedParameters(activeSubmitter.get().getParameters());
                }
                OnetimeJob submitter = PgOnetimeUtils.makeJob(submitterTask, activeBlockUntil.get());
                addedId = bazingaStorage.addOnetimeJob(submitter, ActiveUidDuplicateBehavior.MERGE);

            } else {
                addedId = activeSubmitter.get().getId();
            }
        }
        return addedId;
    }

    public void updateSnapshot(DataApiUserId uid, boolean online) {
        boolean albumsEnabled = DynamicVars.photosliceCreateWithAlbums.get() || (DynamicVars.useMpfsAlbumsFlag.get()
                && mpfsClient.arePhotosliceAlbumsEnabledSafe(uid.forMpfs()));
        updateSnapshot(uid, online, albumsEnabled);
    }

    public void updateSnapshot(DataApiUserId uid, boolean online, boolean albumsEnabled) {
        updateSnapshotWithRetries(uid, online, albumsEnabled);
    }

    public void updateSnapshotAfterAlbumsInit(DataApiUserId uid, ListF<PhotoViewLuceneClusterPojo> clusters) {
        tryUpdateSnapshot(uid, true, true, Option.of(clusters));
    }

    private void updateSnapshotWithRetries(DataApiUserId uid, boolean online, boolean albumsEnabled) {
        try {
            tryUpdateSnapshot(uid, online, albumsEnabled);
        } catch (DatabaseExistsException | NotFoundException | OutdatedChangeException e) {
            logger.warn("Failed to update snapshot: {}, retrying: {}", uid, e);
            try {
                tryUpdateSnapshot(uid, online, albumsEnabled);
            } catch (DatabaseExistsException | NotFoundException | OutdatedChangeException ex) {
                //failing permanently since the other thread seems to handle our changes
                logger.warn("Failed to update snapshot again: {}, failing permanentely: {}", uid, ex);
                throw ex;
            }
        }
    }

    private void tryUpdateSnapshot(DataApiUserId uid, boolean online, boolean albumsEnabled) {
        tryUpdateSnapshot(uid, online, albumsEnabled, Option.empty());
    }

    private void tryUpdateSnapshot(DataApiUserId uid, boolean online, boolean albumsEnabled,
            Option<ListF<PhotoViewLuceneClusterPojo>> givenClusters) {
        if (albumsEnabled && dataApiStorageManager.getDatabaseO(uid).exists(db -> !AlbumsUtils.isWithAlbums(db))) {
            photosliceTransitionManager.initAlbums(uid, this);
            return;
        }

        ListF<PhotoViewLuceneClusterPojo> clusterPojos = givenClusters.getOrElse(
                () -> clusterizerManager.getUserClusters(uid, online));

        Option<Snapshot> dbSnapshotO = dataApiStorageManager.getSnapshotO(uid, META_CONDITION);

        if (dbSnapshotO.exists(snapshot ->
                ClusterId.Format.forLucene(clusterPojos).isIncompatibleTo(ClusterId.Format.ofSnapshot(snapshot))))
        {
            logger.info("Dropping user {} snapshot to reformat cluster ids", uid);
            dropSnapshot(uid);
            tryUpdateSnapshot(uid, online, albumsEnabled);

        } else if (dbSnapshotO.isPresent()) {
            logClustersStatistics(uid, clusterPojos, true);
            updateSnapshot(uid, dbSnapshotO.get(), clusterPojos);

        } else {
            logClustersStatistics(uid, clusterPojos, false);
            storeSnapshot(uid, clusterPojos, albumsEnabled);
        }
    }

    private void logClustersStatistics(DataApiUserId uid, ListF<PhotoViewLuceneClusterPojo> clusterPojos, boolean update) {
        IntSummaryStatistics clusterStatistics = countClusterPhotos(clusterPojos);
        logger.info("Cluster for user {} now has {} clusters with total {} photos (min {}, max {}, avg {}) on {}",
                uid, clusterStatistics.getCount(), clusterStatistics.getSum(), clusterStatistics.getMin(),
                clusterStatistics.getMax(), clusterStatistics.getAverage(), update ? "update" : "create");
    }

    private static IntSummaryStatistics countClusterPhotos(ListF<PhotoViewLuceneClusterPojo> clusterPojos) {
        return clusterPojos.stream()
            .mapToInt(photoViewLuceneClusterPojo -> photoViewLuceneClusterPojo.mergedDocs.size())
            .summaryStatistics();
    }

    private void schedulePlacesUpdate(Database database, ListF<String> geoClusterIds) {
        geoClusterIds.iterator().zipWithIndex().forEachRemaining(clusterIdNumTuple -> schedulePlacesUpdate(database,
                clusterIdNumTuple.get1(), clusterIdNumTuple.get2()));
    }

    private void schedulePlacesUpdate(Database database, String clusterId, int num) {
        Instant notEalierThan = Instant.now().withDurationAdded(SCHEDULING_DELAY,
                num / DynamicVars.placesUpdateTasksPerWindow.get());
        Instant notLaterThan = notEalierThan.withDurationAdded(SCHEDULING_DELAY,
                DynamicVars.placesUpdateWindowMultiplier.get());
        new PgBazingaTaskManager(bazingaStorage).schedule(
                new UpdateClusterPlacesTask(database.uid, database.handleValue(), clusterId),
                Random2.R.nextInstant(notEalierThan, notLaterThan));
    }

    public void updateClusterPlaces(DataApiUserId uid, String handle, String clusterId) {
        GeocoderManager.LocalCache cache = new GeocoderManager.LocalCache();

        AtomicInteger mul = new AtomicInteger();
        Supplier<Integer> nextSleep = () -> Random2.R.nextInt(1000 * Math.max(mul.incrementAndGet(), 15));

        while (true) {
            Option<Snapshot> snapshotO = dataApiStorageManager.getSnapshotO(
                    uid, Option.of(handle), CollectionIdCondition.eq(clusterId));

            if (!snapshotO.isPresent()) {
                logger.warn("Snapshot {} for uid {} suddenly disappeared", handle, uid);
                return;
            }
            Option<DataRecord> indexRecord = dataApiStorageManager.getRecord(
                    uid, handle, new SimpleRecordId(IndexToDeltaMapper.INDEX_COLLECTION_ID, clusterId));

            if (!indexRecord.isPresent()) {
                logger.warn("No cluster {} in {} for uid {}", clusterId, handle, uid);
                return;
            }
            if (indexRecord.exists(r -> r.getRev() > snapshotO.get().database.rev)) {
                logger.info("Outdated index record: {} for uid {}", handle, uid);
                ThreadUtils.sleep(nextSleep.get());
                continue;
            }

            IndexedCluster cluster = IndexToDeltaMapper.retrieveClusters(snapshotO.get()
                    .withRecords(indexRecord.plus(snapshotO.get().records()))).single();
            logger.info("Old places for cluster {} in {} for uid {}: {}", clusterId, handle, uid, cluster.getPlaces());

            IndexedCluster newCluster = geocoderManager.addGeocoderDataCached(cluster, cache);
            logger.info("New places for cluster {} in {} for uid {}: {}", clusterId, handle, uid, newCluster.getPlaces());

            Option<RecordChange> diff = IndexToDeltaMapper.updateClusterIndex(cluster, newCluster);
            if (!diff.isPresent()) {
                logger.info("Diff is empty for cluster {} in {} for uid {}", clusterId, handle, uid);
                return;
            }
            try {
                dataApiStorageManager.storeDelta(uid, snapshotO.get().database.rev, RevisionCheckMode.PER_RECORD, diff);
                return;
            } catch (OutdatedChangeException ex) {
                logger.info("Outdated changes: {} for uid {}", handle, uid);
            } catch (UncategorizedDataAccessException e) {
                logger.info("Lock timeout {} for uid {}", handle, uid);
            }
            // doing this should relax the race in the database
            ThreadUtils.sleep(nextSleep.get());
        }
    }

    public void dropSnapshot(DataApiUserId uid) {
        Option<Database> databaseO = dataApiStorageManager.getDatabaseO(uid);
        if (!databaseO.isPresent()) {
            return;
        }
        dataApiStorageManager.removeDatabase(uid);
        cleanupManager.unregisterDatabase(databaseO.get());
    }

    public void storeSnapshotFromResponseJSONFile(long uid, String filePath, boolean albumsEnabled) {
        PhotoViewLuceneResponsePojo response = PhotoViewCacheBender.searchMapper()
                .parseJson(PhotoViewLuceneResponsePojo.class, InputStreamSourceUtils.valueOf(filePath));
        DataApiPassportUserId user = new DataApiPassportUserId(uid);
        storeSnapshot(user, response.hitsArray, albumsEnabled);
    }

    public void updateSnapshotFromResponseJSONFile(long uid, String filePath) {
        PhotoViewLuceneResponsePojo response = PhotoViewCacheBender.searchMapper()
                .parseJson(PhotoViewLuceneResponsePojo.class, InputStreamSourceUtils.valueOf(filePath));
        updateSnapshot(new DataApiPassportUserId(uid), response.hitsArray);
    }

    private CollectionF<DataRecord> getCollectionRecords(Snapshot metaSnapshot, String collectionId) {
        return metaSnapshot.records().filter(record -> collectionId.equals(record.getCollectionId()));
    }

    private CollectionF<DataRecord> getIndexRecords(Snapshot metaSnapshot) {
        return getCollectionRecords(metaSnapshot, IndexToDeltaMapper.INDEX_COLLECTION_ID);
    }

    private CollectionF<DataRecord> getAlbumRecords(Snapshot metaSnapshot) {
        return getCollectionRecords(metaSnapshot, AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID);
    }
}
