package ru.yandex.chemodan.app.smartcache.worker.processing;

import java.util.NoSuchElementException;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.LockedDatabaseSession;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManagerImpl;
import ru.yandex.chemodan.app.smartcache.AlbumsUtils;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizerManager;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoFields;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoRecordId;
import ru.yandex.chemodan.app.smartcache.worker.processing.tasks.InitAlbumsTask;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.util.sharpei.ShardUserInfo;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class PhotosliceTransitionManager {
    private static final Logger logger = LoggerFactory.getLogger(PhotosliceTransitionManager.class);

    private final BazingaTaskManager bazingaTaskManager;
    private final ClusterizerManager clusterizerManager;
    private final DataApiManager noXivaDataApiManager;
    private final DataRecordsJdbcDao dataRecordsJdbcDao;
    private final DataApiStorageManager dataApiStorageManager;
    private final UserMetaManager userMetaManager;

    private static final SetF<String> SPECIAL_COLLECTION_IDS =
            Cf.set(IndexToDeltaMapper.INDEX_COLLECTION_ID, AlbumsToDeltaMapper.ALBUMS_COLLECTION_ID);

    public void scheduleInitAlbums(DataApiUserId uid) {
        bazingaTaskManager.schedule(new InitAlbumsTask(uid));
    }

    public void initAlbums(DataApiUserId uid, PhotosliceProcessingManager photosliceProcessingManager) {
        try {
            initAlbumsInner(uid, photosliceProcessingManager);
        } catch (NoSuchElementException | IllegalArgumentException e) {
            logger.info("Retrying init after update");
            photosliceProcessingManager.updateSnapshot(uid, true, false);
            initAlbumsInner(uid, photosliceProcessingManager);
        }
    }

    private void initAlbumsInner(DataApiUserId uid, PhotosliceProcessingManager photosliceProcessingManager) {
        if (!dataApiStorageManager.getDatabaseO(uid).isPresent()) {
            return;
        }

        UserDatabaseSpec spec = new UserDatabaseSpec(uid, DataApiStorageManager.PHOTOSLICE_DB_REF);
        ListF<PhotoViewLuceneClusterPojo> clusters = clusterizerManager.getUserClusters(uid, true);

        // Step 1. update all photos records with adding fake empty delta
        // clusters hash is kept same, so it will cause a mismatch on next update
        noXivaDataApiManager.executeWithLockedDatabase(spec, lockedDbSession -> {
            updateUserWithinSession(uid, clusters, lockedDbSession);
            return null;
        });

        // Step 2. all clusters with albums will be updated because of hash mismatch
        // the photos records themselves won't be affected, as they are already updated,
        // but new index and albums records will be created with regular delta
        photosliceProcessingManager.updateSnapshotAfterAlbumsInit(uid, clusters);
    }

    public int getSleepTime(DataApiUserId uid) {
        int shardId = userMetaManager.findMetaUser(uid).map(ShardUserInfo::getShardId).getOrElse(-1);

        return DynamicVars.albumsInitDelayMsByShard.get().map(element -> Cf.x(element.split(":"))).toMap(
                parts -> Integer.parseInt(parts.first()),
                parts -> Integer.parseInt(parts.last())).getOrElse(shardId, DynamicVars.albumsInitDelayMs.get());
    }

    private void updateUserWithinSession(
            DataApiUserId uid, ListF<PhotoViewLuceneClusterPojo> clusters, LockedDatabaseSession lockedDbSession)
    {
        if (AlbumsUtils.isWithAlbums(lockedDbSession.getDb())) {
            return;
        }

        DataApiManagerImpl.ManagerSession dbSession = lockedDbSession.getSessionForHacks();

        Snapshot snapshot = dbSession.getSnapshotO(RecordsFilter.DEFAULT
                .withRecordOrder(ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC)).get();

        MapF<String, ListF<PhotoViewLuceneInfoPojo>> newPhotosByFileId =
                clusters.flatMap(c -> c.mergedDocs).groupBy(p -> p.id);

        ListF<DataRecord> photoRecords = snapshot.records()
                .filterNot(r -> SPECIAL_COLLECTION_IDS.containsTs(r.getCollectionId())).toList();

        Tuple2List<DataRecord, PhotoViewLuceneInfoPojo> photoRecordUpdatesSource = photoRecords.zipWith(r -> {
            String fileId = PhotoRecordId.parse(r.getRecordId()).fileId;
            ListF<PhotoViewLuceneInfoPojo> foundPhotos = newPhotosByFileId.getO(fileId).flatten();

            return foundPhotos.singleO().getOrThrow("No record for ", r.getRecordId());
        });

        ListF<DataRecord> photoRecordUpdates = photoRecordUpdatesSource.map((record, photo) -> {
            Validate.equals(PhotoFields.PATH.get(record), photo.key);
            return record.withData(new DataFields(IndexToDeltaMapper.collectFields(photo)));
        });

        dataRecordsJdbcDao.updateBatched(uid, photoRecordUpdates);

        dbSession.applyDeltas(snapshot.database.withDescription(Option.of(AlbumsUtils.ALBUMS_DESCRIPTION)),
                Cf.list(Delta.empty()), DataApiManagerImpl.RevisionChecker.ifMatch(RecordCondition.all()));
    }
}
