package ru.yandex.chemodan.app.smartcache.worker.processing;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.ClusterizerManager;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.AlbumType;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.DataApiStorageManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumFields;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.PhotoFields;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestsContextConfiguration;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
@ContextConfiguration(classes = {
        TestsContextConfiguration.class
})
@TestExecutionListeners(value = {
        DependencyInjectionTestExecutionListener.class
})
@ActivateDataApiEmbeddedPg
public class PhotosliceTransitionManagerTest extends DataApiTestSupport {

    @Autowired
    private PhotosliceProcessingManager photosliceProcessingManager;
    @Autowired
    private PhotosliceTransitionManager photosliceTransitionManager;
    @Autowired
    private DataApiStorageManager dataApiStorageManager;
    @Autowired
    private MpfsClient mpfsClient;
    @Autowired
    private ClusterizerManager clusterizerManager;
    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    private DataApiUserId testUid;

    @Before
    public void before() {
        testUid = createRandomCleanUserInDefaultShard();
        Mockito.when(mpfsClient.isQuickMoveEnabled(Mockito.anyString())).thenReturn(false);
    }

    @Test
    public void initAlbums() {
        dynamicPropertyManager.setValue(DynamicVars.updateSnapshotDiffBatchSize, 1_000_000);
        Instant date = Instant.parse("2019-11-14T09:06:08.000Z");

        PhotoViewLuceneInfoPojo photo = new PhotoViewLuceneInfoPojo(
                "b25eef7508e634d14ebae1570481adb5968ba070c72e19ad4288f36c1c593e59",
                "/disk/myPhoto.jpg", date, Option.empty(), Option.empty(), Option.of(640), Option.of(480), Option.of(10.0), 1L, Option.of("9"), Option.of(
                AlbumType.CAMERA), Option.empty(), false);

        PhotoViewLuceneClusterPojo cluster = new PhotoViewLuceneClusterPojo(1, date, date, Cf.list(photo));

        Mockito.when(clusterizerManager.getUserClusters(Mockito.any(), Mockito.anyBoolean())).thenReturn(Cf.list(cluster));

        photosliceProcessingManager.handleSnapshotInit(testUid, false, true, false);

        Snapshot snapshotWithoutAlbums = dataApiStorageManager.getSnapshot(testUid, CollectionIdCondition.all());

        photosliceTransitionManager.initAlbums(testUid, photosliceProcessingManager);

        Snapshot snapshotWithAlbums = dataApiStorageManager.getSnapshot(testUid, CollectionIdCondition.all());

        Assert.isEmpty(snapshotWithoutAlbums.database.meta.description);
        Assert.hasSize(2, snapshotWithoutAlbums.records());
        Assert.equals(Cf.set("_index", "0000001573722368000_0000001573722368000"),
                snapshotWithoutAlbums.records().map(DataRecord::getCollectionId).unique());

        DataRecord photoRecord = snapshotWithoutAlbums.records()
                .filter(record -> record.getCollectionId().equals("0000001573722368000_0000001573722368000")).single();
        DataRecord indexRecord = snapshotWithoutAlbums.records()
                .filter(record -> record.getCollectionId().equals("_index")).single();

        Assert.isEmpty(PhotoFields.ALBUMS.getO(photoRecord));
        Assert.isEmpty(indexRecord.getFields().getO("albums_camera"));
        Assert.isEmpty(indexRecord.getFields().getO("albums_beautiful"));

        Assert.some("albums", snapshotWithAlbums.database.meta.description);
        Assert.hasSize(5, snapshotWithAlbums.records());
        Assert.equals(Cf.set("_index", "_albums", "0000001573722368000_0000001573722368000"),
                snapshotWithAlbums.records().map(DataRecord::getCollectionId).unique());


        photoRecord = snapshotWithAlbums.records()
                .filter(record -> record.getCollectionId().equals("0000001573722368000_0000001573722368000")).single();
        indexRecord = snapshotWithAlbums.records()
                .filter(record -> record.getCollectionId().equals("_index")).single();
        ListF<DataRecord> albumsRecords = snapshotWithAlbums.records()
                .filter(record -> record.getCollectionId().equals("_albums")).toList();

        Assert.equals(Cf.set(AlbumType.CAMERA, AlbumType.BEAUTIFUL, AlbumType.NONPHOTOUNLIM),
                PhotoFields.ALBUMS.get(photoRecord));
        Assert.equals(1L, indexRecord.getFields().get("albumCamera").integerValue());
        Assert.equals(1L, indexRecord.getFields().get("albumBeautiful").integerValue());
        Assert.equals(1L, indexRecord.getFields().get("albumNonphotounlim").integerValue());
        Assert.exists(albumsRecords, r -> r.getRecordId().equals("camera") && AlbumFields.COUNT.get(r) == 1);
        Assert.exists(albumsRecords, r -> r.getRecordId().equals("beautiful") && AlbumFields.COUNT.get(r) == 1);
        Assert.exists(albumsRecords, r -> r.getRecordId().equals("nonphotounlim") && AlbumFields.COUNT.get(r) == 1);

        Database database = snapshotWithAlbums.database;

        ListF<Delta> deltas = dataApiStorageManager.getDeltasList(testUid, database.handleValue(), 1, database.rev,
                snapshotWithAlbums.records().map(DataRecord::getCollectionId), 100);

        Assert.hasSize(2, deltas);
        Assert.isEmpty(deltas.first().changes); // first fake delta
        ListF<RecordChange> recordChanges = deltas.last().changes; // second delta with only _index and _albums changes

        Assert.hasSize(4, recordChanges);
        Assert.equals(Cf.set("_index", "_albums"), recordChanges.map(rc -> rc.collectionId).unique());
    }
}
