package ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching;

import java.util.Comparator;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.DiffType;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.MatchedClustersDiff;
import ru.yandex.misc.algo.BinarySearch;

/**
 * @author osidorkin
 */
public class PhotoViewClusterMatcher {
    private static final double MATCHING_THRESHOLD = 0.5;

    public static ListF<ClusterDiff> performMatching(
            ListF<IndexedCluster> oldClusters, ListF<PhotoViewLuceneClusterPojo> newClusters)
    {
        ClusterId.Format fmt = ClusterId.Format.forUpdate(oldClusters, newClusters);

        if (fmt == ClusterId.Format.ADVANCED) {
            newClusters = cutForAdvancedFormat(newClusters);
        }
        if (fmt.isIncompatibleTo(ClusterId.Format.ofSnapshot(oldClusters))) {
            return recreationDiff(oldClusters, newClusters, fmt);
        }

        ListF<ClusterDiff> result = Cf.arrayList();

        Function1V<PhotoViewLuceneClusterPojo> addCluster = c -> result.add(ClusterDiff.clusterAdded(fmt, c));

        int nextStart = 0;

        for (IndexedCluster currentCluster: oldClusters) {
            ListF<PhotoViewLuceneClusterPojo> nextClusters = newClusters.subList(nextStart, newClusters.size());

            int overlapStart = BinarySearch.lowerBound(
                    nextClusters, Comparator.comparing(c -> -c.min.getMillis()),
                    PhotoViewLuceneClusterPojo.empty(currentCluster.getTo()));

            Option<Tuple2<MatchedClustersDiff, Integer>> match = nextClusters
                    .subList(overlapStart, nextClusters.size()).iterator()
                    .takeWhile(cluster -> !isCurrentClusterAfterPojo(currentCluster, cluster))
                    .zipWithIndex()
                    .filterMap(clusterIdx -> matchByPhotos(currentCluster, clusterIdx.get1())
                            .map(diff -> Tuple2.tuple(diff, overlapStart + clusterIdx.get2()))).nextO();

            if (match.isPresent()) {
                int matchIdx = match.get().get2();
                IndexedCluster newCluster = match.get().get1().getNewCluster();

                ClusterId currentId = currentCluster.getId();

                IteratorF<PhotoViewLuceneClusterPojo> addition = nextClusters.subList(0, matchIdx).iterator();
                addition.takeWhile(p -> currentId.getFrom().isBefore(p.min)).forEachRemaining(addCluster);

                if (currentId.getFrom().isAfter(newCluster.getTo())
                        || nextClusters.getO(matchIdx + 1).isMatch(c -> !currentId.isGreater(c.min, c.max)))
                {
                    result.add(ClusterDiff.clusterAdded(newCluster));
                    result.add(ClusterDiff.clusterRemoved(currentCluster));

                } else if (match.get().get1().getType() != DiffType.EXACT_MATCH) {
                    result.add(match.get().get1());
                }
                addition.forEachRemaining(addCluster);

                nextStart += matchIdx + 1;

            } else {
                nextClusters.subList(0, overlapStart).forEach(addCluster);
                result.add(ClusterDiff.clusterRemoved(currentCluster));

                nextStart += overlapStart;
            }
        }
        newClusters.subList(nextStart, newClusters.size()).forEach(addCluster);

        return result;
    }

    static ListF<ClusterDiff> recreationDiff(
            ListF<IndexedCluster> oldClusters, ListF<PhotoViewLuceneClusterPojo> newClusters, ClusterId.Format format)
    {
        return Cf.<ClusterDiff>list()
                .plus(oldClusters.map(ClusterDiff::clusterRemoved))
                .plus(newClusters.map(c -> ClusterDiff.clusterAdded(format, c)));
    }

    static ListF<PhotoViewLuceneClusterPojo> cutForAdvancedFormat(ListF<PhotoViewLuceneClusterPojo> clusters) {
        int from = BinarySearch.lowerBound(
                clusters, Comparator.comparing(c -> -c.min.getMillis()),
                PhotoViewLuceneClusterPojo.empty(new DateTime(9999, 1, 1, 0, 0, DateTimeZone.UTC).toInstant()));

        int to = BinarySearch.upperBound(
                clusters, Comparator.comparing(c -> -c.min.getMillis()),
                PhotoViewLuceneClusterPojo.empty(new DateTime(0, 1, 1, 0, 0, DateTimeZone.UTC).toInstant()));

        return from == 0 && to == clusters.length() ? clusters : clusters.subList(from, to);
    }

    static Option<MatchedClustersDiff> matchByPhotos(
            IndexedCluster currentOldCluster, PhotoViewLuceneClusterPojo candidateClusterPojo)
    {
        if (currentOldCluster.isNotLoaded()
                && currentOldCluster.getPhotosHash().isSome(candidateClusterPojo.getPhotosHash()))
        {
            return Option.of(ClusterDiff.exactMatch(
                    currentOldCluster, candidateClusterPojo, currentOldCluster.getPhotosCount()));
        }

        int commonPhotosCount = countCommonPhotos(currentOldCluster.getPhotos(), candidateClusterPojo.mergedDocs);
        if (commonPhotosCount == currentOldCluster.getPhotosCount()
                && commonPhotosCount == candidateClusterPojo.size)
        {
            if (!currentOldCluster.getPhotosHash().isSome(candidateClusterPojo.getPhotosHash())) {
                return Option.of(ClusterDiff.majorityMatch(currentOldCluster, candidateClusterPojo, commonPhotosCount));
            }
            return Option.of(ClusterDiff.exactMatch(currentOldCluster, candidateClusterPojo, commonPhotosCount));
        }
        if (commonPhotosCount >= MATCHING_THRESHOLD * currentOldCluster.getPhotosCount()
                && commonPhotosCount >= MATCHING_THRESHOLD * candidateClusterPojo.size)
        {
            return Option.of(ClusterDiff.majorityMatch(currentOldCluster, candidateClusterPojo, commonPhotosCount));
        }
        return Option.empty();
    }

    static boolean isCurrentClusterBeforePojo(IndexedCluster currentOldCluster, PhotoViewLuceneClusterPojo clusterPojo) {
        return currentOldCluster.getTo().isBefore(clusterPojo.min);
    }

    static boolean isCurrentClusterAfterPojo(IndexedCluster currentOldCluster, PhotoViewLuceneClusterPojo clusterPojo) {
        return currentOldCluster.getFrom().isAfter(clusterPojo.max);
    }

    //Counts number of photos containing in the both lists
    static int countCommonPhotos(ListF<PhotoViewLuceneInfoPojo> first, ListF<PhotoViewLuceneInfoPojo> second) {
        if (first.size() > second.size()) {
            return countCommonPhotos(Cf.set(second), first);
        }
        return countCommonPhotos(Cf.set(first), second);
    }

    public static int countCommonPhotos(SetF<PhotoViewLuceneInfoPojo> first, ListF<PhotoViewLuceneInfoPojo> second) {
        int count = 0;
        for (PhotoViewLuceneInfoPojo photoViewLuceneInfoPojo: second) {
            if (first.containsTs(photoViewLuceneInfoPojo)) {
                count++;
            }
        }
        return count;
    }

}
