package ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching;


import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneResponsePojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.LocalizedStringDictionary;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.IndexToDeltaMapper;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.DiffType;
import ru.yandex.chemodan.app.smartcache.worker.processing.clusterMatching.diffs.MatchedClustersDiff;
import ru.yandex.chemodan.app.smartcache.worker.tests.TestUtils;
import ru.yandex.chemodan.app.smartcache.worker.utils.PhotoViewCacheBender;
import ru.yandex.chemodan.app.smartcache.worker.utils.PojoConverters;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
public class PhotoViewClusterMatcherTest {

    @Test
    public void testClustersIntersectsByDate() {
        {
            IndexedCluster first = createIndexedCluster(instant(100), instant(200));
            PhotoViewLuceneClusterPojo second = createCluster(instant(300), instant(400));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertTrue(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
        {
            IndexedCluster first = createIndexedCluster(instant(100), instant(300));
            PhotoViewLuceneClusterPojo second = createCluster(instant(200), instant(400));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
        {
            IndexedCluster first = createIndexedCluster(instant(200), instant(300));
            PhotoViewLuceneClusterPojo second = createCluster(instant(100), instant(400));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
        {
            IndexedCluster first = createIndexedCluster(instant(200), instant(400));
            PhotoViewLuceneClusterPojo second = createCluster(instant(100), instant(300));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
        {
            IndexedCluster first = createIndexedCluster(instant(300), instant(400));
            PhotoViewLuceneClusterPojo second = createCluster(instant(100), instant(200));
            Assert.assertTrue(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
        {
            IndexedCluster first = createIndexedCluster(instant(100), instant(400));
            PhotoViewLuceneClusterPojo second = createCluster(instant(200), instant(300));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterAfterPojo(first, second));
            Assert.assertFalse(PhotoViewClusterMatcher.isCurrentClusterBeforePojo(first, second));
        }
    }

    @Test
    public void testCommonPhotosCount() {
        ListF<PhotoViewLuceneInfoPojo> photosPojos = getPhotosPojosList();
        final PhotoViewLuceneInfoPojo photoPojo1 = photosPojos.get(0);
        final PhotoViewLuceneInfoPojo photoPojo2 = photosPojos.get(1);
        final PhotoViewLuceneInfoPojo photoPojo3 = photosPojos.get(2);
        final PhotoViewLuceneInfoPojo photoPojo4 = photosPojos.get(3);
        Assert.assertEquals(0, PhotoViewClusterMatcher.countCommonPhotos(
                Cf.list(photoPojo1, photoPojo2),
                Cf.list(photoPojo3, photoPojo4)));
        Assert.assertEquals(1, PhotoViewClusterMatcher.countCommonPhotos(
                Cf.list(photoPojo1, photoPojo2),
                Cf.list(photoPojo2, photoPojo3, photoPojo4)));
        Assert.assertEquals(2, PhotoViewClusterMatcher.countCommonPhotos(
                Cf.list(photoPojo1, photoPojo2, photoPojo3),
                Cf.list(photoPojo2, photoPojo3, photoPojo4)));
        Assert.assertEquals(4, PhotoViewClusterMatcher.countCommonPhotos(
                Cf.list(photoPojo1, photoPojo2, photoPojo3, photoPojo4),
                Cf.list(photoPojo1, photoPojo2, photoPojo3, photoPojo4)));
    }

    @Test
    public void testClusterDiffs() {
        PhotoViewLuceneResponsePojo response_first = PhotoViewCacheBender.searchMapper()
                .parseJson(PhotoViewLuceneResponsePojo.class,
                        new ClassPathResourceInputStreamSource(getClass(), "response_first.json"));
        PhotoViewLuceneResponsePojo response_second = PhotoViewCacheBender.searchMapper()
                .parseJson(PhotoViewLuceneResponsePojo.class,
                        new ClassPathResourceInputStreamSource(getClass(), "response_second.json"));
        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(response_first.hitsArray);

        ListF<ClusterDiff> diffs = performCheckedMatching(snapshot, response_second.hitsArray);
        Assert.assertEquals(3, diffs.size());
    }

    @Test
    public void testEmptyClusterMatching() {
        ListF<IndexedCluster> snapshot = Cf.list();
        ListF<PhotoViewLuceneInfoPojo> photosPojos = getPhotosPojosList();
        ListF<PhotoViewLuceneClusterPojo> clusters = photosPojos
                .map(pojo -> new PhotoViewLuceneClusterPojo(1, pojo.date, pojo.date, Cf.list(pojo)));
        ListF<ClusterDiff> diffs = PhotoViewClusterMatcher.performMatching(snapshot, clusters);
        Assert.equals(clusters.size(), diffs.size());
        Assert.assertForAll(diffs, (diff) -> { return diff.getType() == DiffType.ADD; });
    }

    @Test
    public void testEmptyResponseMatching() {
        ListF<PhotoViewLuceneInfoPojo> photosPojos = getPhotosPojosList();
        ListF<IndexedCluster> snapshot = photosPojos
                .map(pojo -> new PhotoViewLuceneClusterPojo(1, pojo.date, pojo.date, Cf.list(pojo)))
                .map(PojoConverters::buildClustersIndexStaleFormatted);
        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list();
        ListF<ClusterDiff> diffs = PhotoViewClusterMatcher.performMatching(snapshot, clusters);
        Assert.equals(snapshot.size(), diffs.size());
        Assert.assertForAll(diffs, (diff) -> { return diff.getType() == DiffType.REMOVE; });
    }

    @Test
    public void testPhotoRename() {
        PhotoViewLuceneClusterPojo initialCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto("name1")).reverse());
        IndexedCluster oldCluster = PojoConverters.buildClustersIndexStaleFormatted(initialCluster);
        PhotoViewLuceneClusterPojo updatedCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto("name2")).reverse());

        ListF<ClusterDiff> diffs = performCheckedMatching(Cf.list(oldCluster), Cf.list(updatedCluster));
        Assert.sizeIs(1, diffs);

        Tuple2<ListF<RecordChange>, ListF<RecordChange>> indexAndDataChanges =
                IndexToDeltaMapper.generateClusterDelta(diffs.first()).partition(c -> c.collectionId.equals("_index"));

        Assert.sizeIs(1, indexAndDataChanges.get2());
        Assert.equals(Cf.list("photosHash"), indexAndDataChanges.get1().single().fieldChanges.map(f -> f.key));
    }

    @Test
    public void testPhotoVersionChanges() {
        PhotoViewLuceneClusterPojo initialCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto(12L)).reverse());
        IndexedCluster oldCluster = PojoConverters.buildClustersIndexStaleFormatted(initialCluster);
        PhotoViewLuceneClusterPojo updatedCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto(37L)).reverse());

        ListF<ClusterDiff> diffs = performCheckedMatching(Cf.list(oldCluster), Cf.list(updatedCluster));
        Assert.sizeIs(1, diffs);

        Tuple2<ListF<RecordChange>, ListF<RecordChange>> indexAndDataChanges =
                IndexToDeltaMapper.generateClusterDelta(diffs.first()).partition(c -> c.collectionId.equals("_index"));

        Assert.sizeIs(2, indexAndDataChanges.get2());
        Assert.equals(Cf.list("photosHash"), indexAndDataChanges.get1().single().fieldChanges.map(f -> f.key));
    }

    @Test
    public void testMajorityMatchClusterGeo() {
        PhotoViewLuceneClusterPojo initialCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto("name1")).reverse());
        IndexedCluster oldCluster = PojoConverters.buildClustersIndexStaleFormatted(initialCluster);

        ListF<LocalizedStringDictionary> places = Cf.list("place0", "place1", "place2", "place3")
                .map(TestUtils::createLocalizedStringDictionary);
        Option<LocalizedStringDictionary> localityO = Option.of(new LocalizedStringDictionary(Cf.list("Город", "City")));

        oldCluster = oldCluster.withGeoLocality(localityO).withGeoPlaces(places);

        PhotoViewLuceneClusterPojo updatedCluster = createCluster(
                getCommonPhotos().plus(getLastPhoto("name2")).reverse());
        ListF<ClusterDiff> diffs = performCheckedMatching(Cf.list(oldCluster), Cf.list(updatedCluster));
        Assert.sizeIs(1, diffs);

        ClusterDiff diff = diffs.first();
        Assert.isInstance(diff, MatchedClustersDiff.class);
        MatchedClustersDiff matchedDiff = (MatchedClustersDiff) diff;
        Assert.assertListsEqual(places, matchedDiff.getNewCluster().getPlaces());
        Assert.assertListsEqual(localityO, matchedDiff.getNewCluster().getLocalityO());
    }

    @Test
    public void testClusterSplit() {
        ListF<PhotoViewLuceneInfoPojo> photos = Cf.list(1000L, 2000L, 3000L, 4000L, 5000L).map(this::createPhoto);

        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(Cf.list(createCluster(photos)));

        ListF<ClusterDiff> diffs = PhotoViewClusterMatcher.performMatching(snapshot, Cf.list(
                createCluster(photos.rtake(1)), createCluster(photos.take(4))));

        Assert.equals(Cf.list(DiffType.ADD, DiffType.MAJORITY_MATCH), diffs.map(ClusterDiff::getType));

        diffs = performCheckedMatching(snapshot, Cf.list(
                createCluster(photos.rtake(3)), createCluster(photos.take(2))));

        Assert.equals(Cf.list(DiffType.MAJORITY_MATCH, DiffType.ADD), diffs.map(ClusterDiff::getType));
    }

    @Test
    public void testClusterTransformation() {
        ListF<PhotoViewLuceneInfoPojo> photos = Cf.list(1000L, 2000L, 3000L, 4000L, 5000L, 6000L).map(this::createPhoto);

        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(Cf.list(
                createCluster(photos.rtake(3)), createCluster(photos.take(3))));

        ListF<ClusterDiff> diffs = performCheckedMatching(snapshot, Cf.list(
                createCluster(photos.rtake(1)), createCluster(photos.take(4))));

        Assert.equals(Cf.list(DiffType.REMOVE, DiffType.ADD, DiffType.MAJORITY_MATCH), diffs.map(ClusterDiff::getType));
    }

    @Test
    public void testIdKeptUniqueLeft() {
        ListF<PhotoViewLuceneInfoPojo> photos = Cf.list(1000, 2000, 3000, 4000).map(this::createPhoto);

        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(Cf.list(createCluster(photos)))
                .map(idx -> idx.withId(ClusterId.Format.STALE.cons(instant(1000), instant(1000))));

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                createCluster(photos.rtake(2)), createCluster(photos.firstO()));

        performCheckedMatching(snapshot, clusters);
    }

    @Test
    public void testIdKeptUniqueRight() {
        ListF<PhotoViewLuceneInfoPojo> photos = Cf.list(1000, 2000, 3000, 4000).map(this::createPhoto);

        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(Cf.list(createCluster(photos)))
                .map(idx -> idx.withId(ClusterId.Format.STALE.cons(instant(4000), instant(4000))));

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                createCluster(photos.lastO()), createCluster(photos.take(2)));

        performCheckedMatching(snapshot, clusters);
    }

    @Test
    public void testIdKeptOrdered() {
        ListF<PhotoViewLuceneInfoPojo> photos = Cf.list(3000, 4000).map(this::createPhoto);

        ListF<IndexedCluster> snapshot = PojoConverters.buildIndexStaleFormatted(Cf.list(createCluster(photos)))
                .map(idx -> idx.withId(ClusterId.Format.STALE.cons(instant(1000), instant(4000))));

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                createCluster(photos), createCluster(Cf.list(createPhoto(2000))));

        ListF<ClusterDiff> diffs = performCheckedMatching(snapshot, clusters);
        ListF<String> diffIds = diffs.map(d -> d.getCluster().getIdForDb());

        diffs = diffs
                .plus(snapshot.filterMap(s -> Option.when(!diffIds.containsTs(s.getIdForDb()), ClusterDiff.clusterAdded(s))))
                .filterNot(d -> d.getType() == DiffType.REMOVE);

        Assert.equals(
                diffs.sortedBy(d -> d.getClusterUpdate().getOrElse(d.getCluster()).getFrom()),
                diffs.sortedBy(d -> d.getCluster().getIdForDb()),
                "Result cluster ids expected to be ordered");
    }

    @Test
    public void cutForAdvancedFormat() {
        Function2<Integer, Integer, PhotoViewLuceneClusterPojo> consCluster = (from, to) ->
                new PhotoViewLuceneClusterPojo(0,
                        new DateTime(from, 1, 1, 0, 0, DateTimeZone.UTC).toInstant(),
                        new DateTime(to, 1, 1, 0, 0, DateTimeZone.UTC).toInstant(), Cf.list());

        ListF<PhotoViewLuceneClusterPojo> clusters = Cf.list(
                consCluster.apply(10000, 10001), consCluster.apply(1986, 2000), consCluster.apply(-1, 1));

        Assert.equals(Cf.list(clusters.get(1)), PhotoViewClusterMatcher.cutForAdvancedFormat(clusters));

        clusters = Cf.list(
                consCluster.apply(9999, 10000), consCluster.apply(9999, 9999),
                consCluster.apply(0, 10), consCluster.apply(0, 0));

        Assert.equals(clusters, PhotoViewClusterMatcher.cutForAdvancedFormat(clusters));

        clusters = Cf.list(consCluster.apply(1961, 1986), consCluster.apply(1999, 2005));

        Assert.isTrue(clusters == PhotoViewClusterMatcher.cutForAdvancedFormat(clusters));
    }

    private static ListF<ClusterDiff> performCheckedMatching(
            ListF<IndexedCluster> snapshot, ListF<PhotoViewLuceneClusterPojo> clusters)
    {
        Assert.unique(snapshot.map(IndexedCluster::getIdForDb));

        SetF<String> ids = Cf.toHashSet(snapshot.map(IndexedCluster::getIdForDb));
        ListF<ClusterDiff> diffs = PhotoViewClusterMatcher.performMatching(snapshot, clusters);

        diffs.forEach(diff -> {
            String id = diff.getCluster().getIdForDb();

            if (diff.getType() == DiffType.REMOVE) {
                Assert.isTrue(ids.removeTs(id), "Non existent record removal: ", id);

            } else if (diff.getType() == DiffType.ADD) {
                Assert.isTrue(ids.add(id), "Existent record addition: " + id);

            } else if (diff.getType() == DiffType.EXACT_MATCH) {
                Assert.isFalse(ids.add(id), "Non existent record update: " + id);
            }
        });
        return diffs;
    }

    private static PhotoViewLuceneClusterPojo createCluster(ListF<PhotoViewLuceneInfoPojo> photos) {
        return new PhotoViewLuceneClusterPojo(photos.size(), photos.first().date, photos.last().date, photos);
    }

    private static ListF<PhotoViewLuceneInfoPojo> getCommonPhotos() {
        return Cf.list(
                new PhotoViewLuceneInfoPojo("f236590b7ceaab57541d4e707d25ca91ab6135c150e3bf7c54fd161a45487602",
                        "/disk/Фотокамера/2012-06-06 23-20-58.JPG",
                        instant(1338996058000L), Option.empty(), Option.empty(), 1L),
                new PhotoViewLuceneInfoPojo("b4a173f8e5a084df8294ccb68b3569f5fece40c419ed35df5f1e2ece723b15b0",
                        "/disk/Фотокамера/2012-06-06 23-20-54.JPG",
                        instant(1338996054000L), Option.empty(), Option.empty(), 2L)
               );
    }

    private static PhotoViewLuceneInfoPojo getLastPhoto(String key) {
        return new PhotoViewLuceneInfoPojo("b25eef7508e634d14ebae1570481adb5968ba070c72e19ad4288f36c1c593e59",
                key, instant(1326833018000L), Option.empty(), Option.empty(), 3L);

    }

    private static PhotoViewLuceneInfoPojo getLastPhoto(long version) {
        return new PhotoViewLuceneInfoPojo("b25eef7508e634d14ebae1570481adb5968ba070c72e19ad4288f36c1c593e59",
                "photo.jpg", instant(1326833018000L), Option.empty(), Option.empty(), version);
    }

    private static ListF<PhotoViewLuceneInfoPojo> getPhotosPojosList() {
        return Cf.list(
                new PhotoViewLuceneInfoPojo("020929811ec3edc2a44e4c90338fadf246713e71b448a9c7b4281332cf393204",
                        "/disk/Святое 29-30.11.2014/Разуваем пламя газовой горелкой. Андрей слева.JPG", new Instant(1417807199000L),
                        Option.of(42.310467), Option.of(55.640378), 1L),
                new PhotoViewLuceneInfoPojo("d45b2a49f9a628827939504b10c6a3b7501c6e9a4ae11a5bb964c4708e2d9c97",
                        "/disk/Святое 29-30.11.2014/Гамиль.JPG", new Instant(1417807199000L),
                        Option.of(42.310478), Option.of(55.640719), 2L),
                new PhotoViewLuceneInfoPojo("44ab0a7b494838465c7511a806e79ca7024f3b9dda94da1df49266649b7ab27b",
                        "/disk/Фотокамера/2015-01-20 21-05-30.MP4", new Instant(1421777202000L),
                        Option.empty(), Option.empty(), 3L),
                new PhotoViewLuceneInfoPojo("113b9c6f96157a39b96280b10959ecdacff7be025957d7f50f0af146b2d27690",
                        "/disk/Социальные сети/Vkontakte/Фото со мной/2753165.jpg", new Instant(1176228688000L),
                        Option.empty(), Option.empty(), 4L)
                );
    }

    private PhotoViewLuceneInfoPojo createPhoto(long ts) {
        return new PhotoViewLuceneInfoPojo("" + ts, "" + ts, instant(ts), Option.empty(), Option.empty(), 0L);
    }

    private static Instant instant(long instant) {
        return new Instant(instant);
    }

    private static PhotoViewLuceneClusterPojo createCluster(Instant min, Instant max) {
        return createCluster(min, max, Cf.list());
    }

    private static PhotoViewLuceneClusterPojo createCluster(Instant min, Instant max, ListF<PhotoViewLuceneInfoPojo> mergedDocs) {
        return new PhotoViewLuceneClusterPojo(mergedDocs.size(), min, max, mergedDocs);
    }

    private static IndexedCluster createIndexedCluster(Instant from, Instant to) {
        return createIndexedCluster(from, to, Cf.list());
    }

    private static IndexedCluster createIndexedCluster(Instant from, Instant to, ListF<PhotoViewLuceneInfoPojo> photos) {
        return IndexedCluster.consFromLuceneData(ClusterId.Format.STALE, from, to, photos.size(), Option.empty(), photos);
    }

}
