package ru.yandex.chemodan.app.smartcache.worker.processing.tasks;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupManager;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.CleanupStats;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.DeltaAccessTimeMdao;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.cleanup.TrackedDelta;
import ru.yandex.chemodan.app.smartcache.worker.utils.DynamicVars;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

public class RemoveExpiredSnapshotTask extends CronTask {
    private static final Logger logger = LoggerFactory.getLogger(RemoveExpiredSnapshotTask.class);



    private final CleanupManager cleanupManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaStorage bazingaStorage;
    private final DeltaAccessTimeMdao deltaAccessTimeMdao;

    public RemoveExpiredSnapshotTask(CleanupManager cleanupManager, BazingaTaskManager bazingaTaskManager,
                                     BazingaStorage bazingaStorage, DeltaAccessTimeMdao deltaAccessTimeMdao) {
        this.cleanupManager = cleanupManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaStorage = bazingaStorage;
        this.deltaAccessTimeMdao = deltaAccessTimeMdao;
    }

    @Override
    public Schedule cronExpression() {
        return ScheduleCron.parse("0 1 * * *", TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    @Override
    public void execute(ExecutionContext executionContext) {

        if (!isAllowed()) {
            logger.info("Cleaning forbidden from {} to {}",
                    DynamicVars.allowRemovingExpiredSnapshotBefore.get(),
                    DynamicVars.allowRemovingExpiredSnapshotAfter.get());
            return;
        }

        if (DynamicVars.useCleaningOnOnetimeTasks.get()) {
            scheduleCleaningTasks();
            return;
        }

        CleanupStats stats = DynamicVars.parallelRemoveExpiredSnapshot.get()
                ? cleanupManager.removeExpiredInParallel()
                : cleanupManager.removeExpired();
        executionContext.setExecutionInfo(stats);
    }

    private boolean isAllowed() {
        return ru.yandex.chemodan.util.TimeUtils.nowBetween(DynamicVars.allowRemovingExpiredSnapshotAfter.get(),
                DynamicVars.allowRemovingExpiredSnapshotBefore.get());
    }


    private void scheduleCleaningTasks() {
        TaskId taskId = TaskId.from(DeleteDeltaTask.class);
        int jobCount = bazingaStorage.findOnetimeJobCount(taskId, JobStatus.READY);
        if (jobCount > DynamicVars.cleaningMinQueueSize.get()) {
            return;
        }

        Instant instant = CleanupManager.getExpirationInstant();
        int limit = DynamicVars.cleaningMongoBatchSize.get();
        int first = 0;
        ListF<TrackedDelta> trackedDeltas;

        do {
            trackedDeltas = deltaAccessTimeMdao.getTrackedItemsBefore(instant, first, limit);
            for (TrackedDelta trackedDelta : trackedDeltas) {
                bazingaTaskManager.schedule(new DeleteDeltaTask(new DeleteDeltaParameters(trackedDelta.deltaId)));
            }

            first += limit;
            jobCount = bazingaStorage.findOnetimeJobCount(taskId, JobStatus.READY);
        } while (jobCount < DynamicVars.cleaningMaxQueueSize.get() && trackedDeltas.isNotEmpty());
    }

}
