package ru.yandex.chemodan.app.smartcache.worker.utils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;

/**
 * @author osidorkin
 * Implements Hirschberg algorithm
 * TODO make it memory-linear
 */
public class ListDiffCalculator {
    public static class Diff<T> {
        private final int position;

        public Diff(int position) {
            this.position = position;
        }

        public int getPosition() {
            return position;
        }
    }

    public static class RemoveDiff<T> extends Diff<T> {
        RemoveDiff(int position) {
            super(position);
        }

        @Override
        public String toString() {
            return "Remove " + getPosition();
        }
    }

    public abstract static class AbstractValueDiff<T> extends Diff<T> {
        private final T value;

        AbstractValueDiff(int position, T value) {
            super(position);
            this.value = value;
        }

        public T getValue() {
            return value;
        }
    }

    public static class InsertDiff<T> extends AbstractValueDiff<T> {
        InsertDiff(int position, T value) {
            super(position, value);
        }

        @Override
        public String toString() {
            return "InsertDiff " + getPosition() + " " + getValue();
        }
    }

    public static class ReplaceDiff<T> extends AbstractValueDiff<T> {
        ReplaceDiff(int position, T value) {
            super(position, value);
        }

        @Override
        public String toString() {
            return "ReplaceDiff " + getPosition() + " " + getValue();
        }
    }

    public static <T> ListF<T> applyDiff(ListF<T> oldList, ListF<Diff<T>> difference) {
        int currentTextPosition = 0;
        int offsetDelta = 0; //difference between removes and inserts
        ListF<T> res = Cf.arrayList();
        for (Diff<T> next: difference) {
            int position = next.getPosition() + offsetDelta;
            if (position >  currentTextPosition) {
                ListF<T> toCopy = oldList.subList(currentTextPosition, position);
                res.addAll(toCopy);
                currentTextPosition = position;

            }
            if (next instanceof ListDiffCalculator.InsertDiff) {
                res.add(((ListDiffCalculator.InsertDiff<T>) next).getValue());
                offsetDelta--;
            } else {
                currentTextPosition++;
                if (next instanceof ListDiffCalculator.RemoveDiff) {
                    offsetDelta++;
                } else if (next instanceof ListDiffCalculator.ReplaceDiff) {
                    res.add(((ListDiffCalculator.ReplaceDiff<T>) next).getValue());
                }
            }
        }

        if (currentTextPosition < oldList.size()) {
            res.addAll(oldList.subList(currentTextPosition, oldList.size()));
        }
        return res;
    }


    public static <T> ListF<Diff<T>> calculateDifference(ListF<T> oldList, ListF<T> newList) {
        PositionItem[][] fullTable = calculateTable(oldList, newList);

        ListF<Diff<T>> res = Cf.arrayList();
        int i = fullTable.length - 1;
        int j = fullTable[0].length - 1;
        while (i > 0 || j > 0) {
            PositionItem item = fullTable[i][j];
            if (item.fromLeftTop) {
                if (fullTable[i - 1][j - 1].weight != item.weight) {
                    res.add(new ReplaceDiff<T>(j - 1, newList.get(j - 1)));
                }
                i--;
                j--;
            } else if (item.fromTop) {
                res.add(new RemoveDiff<T>(j));
                i--;
            } else if (item.fromLeft) {
                res.add(new InsertDiff<T>(j - 1, newList.get(j - 1)));
                j--;
            } else {
                throw new RuntimeException("Nowhere to move?");
            }
        }
        return res.reverse();
    }

    private static final class PositionItem {
        public int weight = 0;
        public boolean fromTop = false;
        public boolean fromLeft = false;
        public boolean fromLeftTop = false;
    }

    private static <T> PositionItem[][] calculateTable(ListF<T> oldList, ListF<T> newList) {
        PositionItem[][] res = new PositionItem[oldList.size() + 1][];
        for (int i = 0; i < res.length; i++) {
            res[i] = new PositionItem[newList.size() + 1];
            res[i][0] = new PositionItem();
            res[i][0].weight = i;
            res[i][0].fromTop = true;
        }

        for (int i = 0; i < res[0].length; i++) {
            res[0][i] = new PositionItem();
            res[0][i].weight = i;
            res[0][i].fromLeft = true;
        }

        for (int i = 0; i < oldList.size(); i++) {
            // here indices are shifted by one in array - 0 is an empty string, 1 - first character
            // but in lists the numeration is not shifted
            for (int j = 0; j < newList.size(); j++) {
                res[i + 1][j + 1] = new PositionItem();
                if (newList.get(j).equals(oldList.get(i))) {
                    res[i + 1][j + 1].weight = res[i][j].weight;
                    res[i + 1][j + 1].fromLeftTop = true;
                } else {
                    int minWeight = Math.min(Math.min(res[i][j + 1].weight, res[i][j].weight), res[i + 1][j].weight);
                    if (res[i][j + 1].weight == minWeight) {
                        res[i + 1][j + 1].fromTop = true;
                    }
                    if (res[i][j].weight == minWeight) {
                        res[i + 1][j + 1].fromLeftTop = true;
                    }
                    if (res[i + 1][j].weight == minWeight) {
                        res[i + 1][j + 1].fromLeft = true;
                    }
                    res[i + 1][j + 1].weight = minWeight + 1;
                }
            }
        }

        return res;
    }
}
