package ru.yandex.chemodan.app.smartcache.worker.utils;

import java.nio.ByteBuffer;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneClusterPojo;
import ru.yandex.chemodan.app.smartcache.worker.clusterizer.pojo.PhotoViewLuceneInfoPojo;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.ClusterId;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.IndexedCluster;
import ru.yandex.chemodan.app.smartcache.worker.dataapi.mappers.AlbumsBitMaskUtils;
import ru.yandex.misc.digest.Digest;
import ru.yandex.misc.digest.Sha1;

/**
 * @author osidorkin
 */
public class PojoConverters {

    public static ListF<IndexedCluster> buildIndex(
            ClusterId.Format format, ListF<PhotoViewLuceneClusterPojo> clusterPojos)
    {
        return clusterPojos.map(c -> buildClustersIndex(format, c));
    }

    public static IndexedCluster buildClustersIndex(ClusterId.Format format, PhotoViewLuceneClusterPojo clusterPojo) {
        return IndexedCluster.consFromLuceneData(
                format, clusterPojo.min, clusterPojo.max,
                clusterPojo.size, Option.of(clusterPojo.getPhotosHash()), clusterPojo.mergedDocs);
    }

    public static ListF<IndexedCluster> buildIndexStaleFormatted(ListF<PhotoViewLuceneClusterPojo> clusterPojos) {
        return buildIndex(ClusterId.Format.STALE, clusterPojos);
    }

    public static IndexedCluster buildClustersIndexStaleFormatted(PhotoViewLuceneClusterPojo clusterPojo) {
        return buildClustersIndex(ClusterId.Format.STALE, clusterPojo);
    }

    public static String computePhotosHash(ListF<PhotoViewLuceneInfoPojo> photos) {
        Digest.Stream stream = Sha1.A.stream();
        ByteBuffer numsBuffer = ByteBuffer.allocate(48);
        ByteBuffer bitMaskBuffer = ByteBuffer.allocate(8); // will be joint with numsBuffer after migration

        photos.forEach(photo -> {
            stream.update(ByteBuffer.wrap(photo.id.getBytes()));
            stream.update(ByteBuffer.wrap(photo.key.getBytes()));

            numsBuffer.clear();
            numsBuffer.putLong(0, photo.date.getMillis());
            numsBuffer.putLong(8, photo.version);

            numsBuffer.putDouble(16, photo.latitude.getOrElse(0d));
            numsBuffer.putDouble(24, photo.longitude.getOrElse(0d));
            numsBuffer.putDouble(32, photo.beauty.getOrElse(0d));
            numsBuffer.putInt(40, photo.width.getOrElse(0));
            numsBuffer.putInt(44, photo.height.getOrElse(0));

            stream.update(numsBuffer);

            if (photo.isWithAlbums()) {
                bitMaskBuffer.clear();
                bitMaskBuffer.putLong(0, AlbumsBitMaskUtils.getBitMask(photo.getAlbums()));
                stream.update(bitMaskBuffer);
            }

        });
        return stream.end().base64();
    }
}
