package ru.yandex.chemodan.alice;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Map;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Struct;
import com.google.protobuf.util.JsonFormat;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.FileTemplateLoader;
import freemarker.core.JSONOutputFormat;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateExceptionHandler;
import org.joda.time.Instant;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Copy-paste from ru.yandex.chemodan.app.hackathon. DivTemplateProcessor without hackaton hacks
 */
public class DivTemplateProcessor {
    private static final Logger logger = LoggerFactory.getLogger(DivTemplateProcessor.class);

    private Template template;
    private Instant lastUpdateTime;

    private Configuration fromFileConfig = null;
    private final File2 templatesBaseDir = new File2("/tmp/htemplates/");
    private final String templateFileName;

    public DivTemplateProcessor(Class<?> resourceLoaderClass, String filename) {
        this.templateFileName = filename;

        Configuration cfg = new Configuration(Configuration.VERSION_2_3_28);

        cfg.setTemplateLoader(new ClassTemplateLoader(resourceLoaderClass, ""));

        cfg.setDefaultEncoding("UTF-8");
        cfg.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
        cfg.setLogTemplateExceptions(true);
        cfg.setWrapUncheckedExceptions(true);
        cfg.setOutputFormat(JSONOutputFormat.INSTANCE);
        cfg.setAutoEscapingPolicy(Configuration.DISABLE_AUTO_ESCAPING_POLICY);

        try {
            template = cfg.getTemplate(filename);
            lastUpdateTime = Instant.now();
        } catch (IOException e) {
            logger.error(e);
            throw ExceptionUtils.translate(e);
        }

        fromFileConfig = new Configuration(Configuration.VERSION_2_3_28);

        initFileConfig();
    }

    private void initFileConfig() {
        if (templatesBaseDir.exists()) {
            try {
                fromFileConfig.setTemplateLoader(new FileTemplateLoader(templatesBaseDir.getFile()));
            } catch (IOException e) {
                logger.error(e);
                throw ExceptionUtils.translate(e);
            }

            fromFileConfig.setDefaultEncoding("UTF-8");
            fromFileConfig.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
            fromFileConfig.setLogTemplateExceptions(true);
            fromFileConfig.setWrapUncheckedExceptions(true);
            fromFileConfig.setOutputFormat(JSONOutputFormat.INSTANCE);
            fromFileConfig.setAutoEscapingPolicy(Configuration.DISABLE_AUTO_ESCAPING_POLICY);
        }
    }

    public Struct processTemplate(Map<String, Object> model) {
        updateTemplateIfNecessary();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Writer out = new OutputStreamWriter(baos);

        try {
            template.process(model, out);
        } catch (Exception e) {
            logger.error(e);
            throw ExceptionUtils.translate(e);
        }

        try {
            Struct.Builder builder = Struct.newBuilder();
            JsonFormat.parser().merge(baos.toString(), builder);
            return builder.build();
        } catch (InvalidProtocolBufferException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public void updateTemplateIfNecessary() {
        initFileConfig();
        if (fromFileConfig == null) {
            return;
        }

        try {
            File2 templateFile = templatesBaseDir.child(templateFileName);
            if (templateFile.exists() && templateFile.lastModified().isAfter(lastUpdateTime)) {
                template = fromFileConfig.getTemplate(templateFileName);
                lastUpdateTime = Instant.now();
                logger.info("Template reloaded from file");
            }
        } catch (Exception e) {
            logger.error("Failed to update template : {}", e);
        }
    }
}

