package ru.yandex.chemodan.bazinga;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.bazinga.impl.worker.BazingaWorkerAvailableMemoryProvider;
import ru.yandex.commune.bazinga.impl.worker.BazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.impl.worker.OnetimeTaskLogConfiguration;
import ru.yandex.commune.bazinga.impl.worker.WorkerLogConfiguration;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.bazinga.scheduler.TaskQueue;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.exec.ProcessUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.os.CpuInfoUtils;

/**
 * @author dbrylev
 */
@Configuration
@Import({
        BazingaConfigurationContextConfiguration.class,
})
public class BazingaWorkerConfigurationContextConfiguration {

    @Bean
    public UntraceableTasks untraceableTasks() {
        return UntraceableTasks.traceNone();
    }

    @Bean
    public BazingaWorkerConfiguration bazingaWorkerConfiguration(
            @Value("${bazinga.log.dir}")
            File2 bazingaLogDir,
            @Value("${bazinga.log.onetime.size-limit}")
            DataSize onetimeLogFileSizeLimit,
            @Value("${bazinga.log.onetime.history-count}")
            int onetimeLogFileHistoryCount,
            @Value("${bazinga.log.onetime.too-old-timeout}")
            Duration onetimeLogTooOldTimeout,

            @Value("${bazinga.log.cron.dir.size.limit}")
            DataSize cronLogDirSizeLimit,
            @Value("${bazinga.log.cron.dir.count.limit}")
            int cronLogDirFilesCountLimit,
            @Value("${bazinga.log.cron.task.size.limit}")
            DataSize cronTaskLogSizeLimit,
            @Value("${bazinga.log.cron.task.count.limit}")
            int cronTaskLogsCountLimit,

            @Value("${bazinga.worker.cpu.intensive.tasks.limit:-}")
            String cpuIntesiveTasksLimitOverride,
            @Value("${bazinga.worker.regular.tasks.limit}")
            int regularTasksLimit,
            @Value("${bazinga.worker.regular.tasks.queue.size.coeff}")
            int reqularTaskCoeff,
            @Value("${bazinga.worker.cpuIntensive.tasks.queue.size.coeff}")
            int cpuIntensiveTaskCoeff,
            BazingaWorkerTaskQueues taskQueues,
            UntraceableTasks untraceableTasks)
    {
        int cpuIntensiveTaskLimit = StringUtils.notBlankO(cpuIntesiveTasksLimitOverride).map(Cf.Integer::parse)
                .getOrElse(CpuInfoUtils.getTotalCoreCount());

        ListF<TaskQueue> queues = Cf.arrayList();

        queues.addAll(taskQueues.cronQueue.map(q ->
                new TaskQueue(q, regularTasksLimit, 100500)));
        queues.addAll(taskQueues.regularQueue.map(q ->
                new TaskQueue(q, regularTasksLimit, regularTasksLimit * reqularTaskCoeff)));
        queues.addAll(taskQueues.cpuIntensiveQueue.map(q ->
                new TaskQueue(q, cpuIntensiveTaskLimit, cpuIntensiveTaskLimit * cpuIntensiveTaskCoeff)));

        queues.addAll(taskQueues.specificQueues);

        return new BazingaWorkerConfiguration(
                Cf.list(TaskCategory.DEFAULT),
                new WorkerLogConfiguration(
                        bazingaLogDir,
                        Option.of(cronLogDirSizeLimit),
                        Option.of(cronLogDirFilesCountLimit),
                        Option.of(cronTaskLogSizeLimit),
                        Option.of(cronTaskLogsCountLimit),
                        Option.of(new OnetimeTaskLogConfiguration(
                                onetimeLogFileSizeLimit,
                                onetimeLogFileHistoryCount,
                                onetimeLogTooOldTimeout,
                                new Duration(0)))),
                Option.empty(),
                queues,
                BazingaWorkerAvailableMemoryProvider.DEFAULT,
                ProcessUtils::destroySafe,
                api -> new UntraceableTasksControllerApi(api, untraceableTasks));
    }
}
