package ru.yandex.chemodan.bazinga;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfigurator;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfiguratorContextConfiguration;
import ru.yandex.chemodan.util.jdbc.logging.QueryInterceptors;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.bazinga.pg.context.PgBazingaStorageContextConfiguration;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorageConfiguration;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsPartitionShardResolver;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsPartitioningConfiguration;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsRelocationManager;
import ru.yandex.commune.bazinga.pg.storage.shard.ShardedDataSource;
import ru.yandex.commune.db.shard.ShardSource;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.admin.conductor.ConductorContextConfiguration;

/**
 * @author dbrylev
 */
@Configuration
@Import({
        PgBazingaStorageContextConfiguration.class,
        JobsCleaningRegistryContextConfiguration.class,
        ConductorContextConfiguration.class,
        JdbcDatabaseConfiguratorContextConfiguration.class,
        DataSourceBazingaStorageContextConfiguration.class,
})
public class PgBazingaStorageConfigurationContextConfiguration {

    private final DynamicProperty<Integer> unfinishedPartitionRelocationRetryDelay = new DynamicProperty<>(
            "bazinga.relocation.unfinished-retry-delay-ms", 500);

    @Autowired
    private JdbcDatabaseConfiguratorContextConfiguration dbConfiguratorConfig;

    @Bean
    public PgBazingaStorageConfiguration bazingaStorageConfiguration(
            @Value("${bazinga.pg.shard.default-number}") int defaultShard,
            @Value("${bazinga.pg.jobs-partitioning.number-of-partitions}") int numberOfPartitions,
            @Value("${bazinga.pg.jobs-partitioning.update-delay}") Duration updateDelay,
            @Value("${bazinga.pg.jobs-partitioning.update-error-delay}") Duration updateErrorDelay,
            @Value("${bazinga.pg.jobs-dao.has-execution-info}") boolean onetimeJobHasExecutionInfo,
            ShardSource bazingaDataSource)
    {
        JobsPartitioningConfiguration partitioning = new JobsPartitioningConfiguration(
                numberOfPartitions, updateDelay, updateErrorDelay);

        return new PgBazingaStorageConfiguration(defaultShard, bazingaDataSource,
                Option.of(QueryInterceptors::defaultQueryInterceptor), partitioning,
                () -> Duration.millis(unfinishedPartitionRelocationRetryDelay.get()),
                onetimeJobHasExecutionInfo);
    }

    @Bean
    public ShardSource bazingaDataSource(JdbcDatabaseConfigurator bazingaDbConfigurator) {
        return bazingaDbConfigurator.configureSharded(d -> { throw new UnsupportedOperationException(); });
    }

    @Bean
    public JobsRelocationManager jobsRelocationManager(PgBazingaStorageConfiguration conf) {
        return Option.of(conf.dataSource).filterByType(ShardedDataSource.class)
                .map(x -> new JobsRelocationManager(x, conf.onetimeJobHasExecutionInfo)).getOrNull();
    }

    @Bean
    public PingerChecker bazingaPartitionsPingerChecker(JobsPartitionShardResolver resolver) {
        return resolver::isInitialized;
    }
}
