package ru.yandex.chemodan.bazinga;

import java.io.IOException;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Wrapper;
import java.util.logging.Logger;

import javax.sql.CommonDataSource;
import javax.sql.DataSource;

import lombok.AllArgsConstructor;
import lombok.experimental.Delegate;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.jdbc.DcAwareDynamicMasterSlaveDataSource;
import ru.yandex.chemodan.util.jdbc.logging.LastAccessedDsSource;
import ru.yandex.commune.db.shard.ShardInfo;
import ru.yandex.commune.db.shard.proxy.Discriminant;
import ru.yandex.commune.db.shard.spring.ShardResolver;
import ru.yandex.commune.db.shard2.Shard2;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.db.shard2.ShardMetaNotifier;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.db.ExposeConnectionCountDataSource;
import ru.yandex.misc.thread.ThreadUtils;

public class ShardSourceWithMalfunctionEmulator extends ShardManager2 {

    private DynamicProperty<ListF<String>> shardMalfunctions =
            DynamicProperty.cons("sharded-datasource-malfunction-emulation", Cf.list());

    private final ShardManager2 shardManager;

    public ShardSourceWithMalfunctionEmulator(ShardManager2 shardManager) {
        // super isn't actually used. just need it to be a proper ShardManager object
        super(Mockito.mock(ShardResolver.class), Mockito.mock(ShardMetaNotifier.class), null, null, null);
        this.shardManager = shardManager;
    }

    @Override
    public Tuple2List<ShardInfo, DataSource> getShardInfoAndDataSourceList() {
        return shardManager.getShardInfoAndDataSourceList()
                .map((info, ds) -> Tuple2.tuple(info, new DataSourceWithMalfunction(
                        info.getId(), (DcAwareDynamicMasterSlaveDataSource) ds)))
                .toTuple2List(t -> t._1, t -> t._2);
    }

    @Override
    public Shard2 getShard(int shardId) {
        return shardManager.getShard(shardId);
    }

    @Override
    public Shard2 getShardByDiscriminant(Discriminant d) {
        return shardManager.getShardByDiscriminant(d);
    }

    @Override
    public CollectionF<Shard2> shards() {
        return shardManager.shards();
    }

    @Override
    public CollectionF<Shard2> getWritableShards() {
        return shardManager.getWritableShards();
    }

    @Override
    public void init() {
        shardManager.init();
    }

    @Override
    public void close() throws IOException {
        shardManager.close();
    }

    @Override
    public void destroy() {
        shardManager.destroy();
    }

    private void sleepOrFail(int shardId) {
        long delay = shardMalfunctions.get().map(element -> Cf.x(element.split(":"))).toMap(
                parts -> Integer.parseInt(parts.first()),
                parts -> Long.parseLong(parts.last()))
                .getOrElse(shardId, 0L);
        if (delay < 0) {
            throw new RuntimeException("Shard " + shardId + " failure emulation");
        }
        ThreadUtils.sleep(delay);
    }

    @AllArgsConstructor
    private class DataSourceWithMalfunction implements DataSource, LastAccessedDsSource {
        private final int shardId;
        @Delegate(
                types = {CommonDataSource.class, Wrapper.class,
                        LastAccessedDsSource.class, ExposeConnectionCountDataSource.class},
                excludes = {DataSource.class})
        private final DcAwareDynamicMasterSlaveDataSource dataSource;

        @Override
        public Connection getConnection() throws SQLException {
            sleepOrFail(shardId);
            return dataSource.getConnection();
        }

        @Override
        public Connection getConnection(String username, String password) throws SQLException {
            sleepOrFail(shardId);
            return dataSource.getConnection(username, password);
        }

        @Override
        public <T> T unwrap(Class<T> iface) throws SQLException {
            return dataSource.unwrap(iface);
        }

        @Override
        public boolean isWrapperFor(Class<?> iface) throws SQLException {
            return dataSource.isWrapperFor(iface);
        }

        @Override
        public PrintWriter getLogWriter() throws SQLException {
            return dataSource.getLogWriter();
        }

        @Override
        public void setLogWriter(PrintWriter out) throws SQLException {
            dataSource.setLogWriter(out);
        }

        @Override
        public void setLoginTimeout(int seconds) throws SQLException {
            dataSource.setLoginTimeout(seconds);
        }

        @Override
        public int getLoginTimeout() throws SQLException {
            return dataSource.getLoginTimeout();
        }

        @Override
        public Logger getParentLogger() throws SQLFeatureNotSupportedException {
            return dataSource.getParentLogger();
        }
    }
}
