package ru.yandex.chemodan.bazinga.dynamic;

import java.io.PrintWriter;
import java.io.StringWriter;

import org.joda.time.Duration;
import org.springframework.context.ApplicationContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRegistry;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.bazinga.scheduler.schedule.SchedulePeriodic;
import ru.yandex.commune.script.cosher.InterpreterPopulator;
import ru.yandex.commune.script.cosher.InterpreterUtils;
import ru.yandex.commune.script.cosher.RichInterpreter;
import ru.yandex.commune.script.cosher.nashorn.NashornInterpreter;
import ru.yandex.commune.script.cosher.nashorn.NashornInterpreterDefaultPopulator;
import ru.yandex.commune.util.serialize.ToMultilineSerializer;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class DynamicCronTasksManager extends ZkRegistry<String, DynamicCronTaskInfo> {
    private static final Logger logger = LoggerFactory.getLogger(DynamicCronTasksManager.class);

    private static final String PREFIX = "dynamic_";

    private final BazingaWorkerApp bazingaWorkerApp;
    private final TaskQueueName cronTaskQueue;

    private final ApplicationContext applicationContext;
    private final ToMultilineSerializer toMultilineSerializer;

    public DynamicCronTasksManager(
            ZkPath zkPath,
            BazingaWorkerApp bazingaWorkerApp, TaskQueueName cronTaskQueue,
            ApplicationContext applicationContext,
            ToMultilineSerializer toMultilineSerializer)
    {
        super(zkPath, BazingaBender.mapper.createParserSerializer(DynamicCronTaskInfo.class), task -> task.id, s -> s);
        this.bazingaWorkerApp = bazingaWorkerApp;
        this.cronTaskQueue = cronTaskQueue;
        this.applicationContext = applicationContext;
        this.toMultilineSerializer = toMultilineSerializer;

        super.addListener(s -> updateTasks());
    }

    public void updateTasks() {
        ListF<CronTask> dynTasks = getAll().map(this::consTask).filter(q -> q.queueName().equals(cronTaskQueue));
        WorkerTaskRegistry taskRegistry = bazingaWorkerApp.getWorkerTaskRegistry();

        ListF<CronTask> existingTasks = taskRegistry
                .getCronTasks()
                .filter(task -> task.id().toString().startsWith(PREFIX));

        taskRegistry.addTasks(dynTasks.toMapMappingToKey(CronTask::id), Cf.map(), Cf.map());

        ListF<TaskId> toRemove = existingTasks.map(CronTask::id).unique().minus(dynTasks.map(CronTask::id)).toList();
        bazingaWorkerApp.getBazingaWorker().suspendTaskScheduling(toRemove);
        bazingaWorkerApp.getBazingaWorker().resumeTaskScheduling(dynTasks, Cf.list());
    }

    private CronTask consTask(DynamicCronTaskInfo info) {
        return new CronTask() {
            @Override
            public TaskId id() {
                if (queueName().equals(super.queueName())) {
                    return new TaskId(PREFIX + info.id);
                } else  {
                    String queuePrefix = queueName().getName()
                            .replace("cron", "")
                            .replace("-", "");
                    return new TaskId(PREFIX + queuePrefix + "_" + info.id);
                }
            }

            @Override
            public TaskQueueName queueName() {
                return info.queueName.map(TaskQueueName::new).getOrElse(super.queueName());
            }

            @Override
            public Schedule cronExpression() {
                switch (info.scheduleType) {
                    case CRON:
                        return ScheduleCron.parse(info.scheduleExpression, TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
                    case PERIOD:
                        return new SchedulePeriodic(Duration.standardSeconds(Long.parseLong(info.scheduleExpression)));
                    default:
                        return new SchedulePeriodic(Duration.standardDays(1));
                }
            }

            @Override
            public void execute(ExecutionContext executionContext) throws Exception {
                InterpreterPopulator interpreterPopulator = new NashornInterpreterDefaultPopulator();

                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw, true);
                RichInterpreter rich = new NashornInterpreter(pw);

                InterpreterUtils.loadContext(rich, applicationContext);
                interpreterPopulator.populate(rich);

                rich.bindVariable("executionContext", executionContext);

                Option<Object> result = Option.empty();
                try {
                    result = Option.of(rich.eval(info.scriptText));

                } finally {
                    if (sw.getBuffer().length() > 0) {
                        logger.info("Script output: {}", sw);
                    }
                    result.filterNotNull().forEach(r ->
                        logger.info("Script result: {}", toMultilineSerializer.serialize(r)));
                }
            }
        };
    }

    public void save(DynamicCronTaskInfo dynamicCronTaskInfo) {
        put(dynamicCronTaskInfo);
    }

    public ListF<DynamicCronTaskInfo> listDynamicCronTasks() {
        return getAll().toList();
    }
}
