package ru.yandex.chemodan.bazinga.http;

import java.io.Closeable;
import java.io.IOException;

import org.apache.commons.lang3.NotImplementedException;
import org.apache.http.HttpEntity;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author dbrylev
 */
public class PgHttpBazingaTaskManager implements BazingaTaskManager, Closeable {

    private final HostPort hostPort;
    private final HttpClient httpClient;

    public PgHttpBazingaTaskManager(HostPort hostPort, HttpClient httpClient) {
        this.hostPort = hostPort;
        this.httpClient = httpClient;
    }

    @Override
    public FullJobId schedule(OnetimeTask task) {
        return schedule(task, Instant.now());
    }

    @Override
    public FullJobId schedule(OnetimeTask task, Instant date) {
        return schedule(task, date, task.priority());
    }

    @Override
    public FullJobId schedule(OnetimeTask task, TaskCategory category, Instant date) {
        return schedule(task, category, date, task.priority());
    }

    @Override
    public FullJobId schedule(OnetimeTask task, TaskCategory category, Instant date, int priority) {
        return schedule(task, category, date, priority, false);
    }

    @Override
    public FullJobId schedule(OnetimeTask task, TaskCategory category, Instant date, int priority,
            boolean forceRandomInId)
    {
        return schedule(task, category, date, priority, forceRandomInId, Option.empty());
    }

    @Override
    public FullJobId schedule(OnetimeTask task, TaskCategory category, Instant date, int priority,
            boolean forceRandomInId, Option<String> group)
    {
        return schedule(task, date, priority);
    }

    public FullJobId schedule(OnetimeTask task, Instant date, int priority) {
        return schedule(task, Option.empty(), date, priority);
    }

    @Override
    public FullJobId schedule(OnetimeTask task, TaskCategory category, Instant date, int priority,
            boolean forceRandomInId, Option<String> group, JobId jobId)
    {
        return schedule(task, date, priority);
    }

    public FullJobId schedule(OnetimeTask task, Option<String> activeUniqueIdentifier, Instant date, int priority) {
        PgTasksActionContainer.AddTasksRequest taskRequest = new PgTasksActionContainer.AddTasksRequest(
                task.id().getId(),
                new String(BazingaBender.mapper.serializeJson(task.getParameters())),
                Option.of(priority), Option.of(date), activeUniqueIdentifier);

        HttpPost post = new HttpPost(actionUrl("add-tasks"));
        post.setEntity(serializeRequestEntity(new PgTasksActionContainer.AddTasksRequestList(Cf.list(taskRequest))));

        return executeAndParse(post, PgTasksActionContainer.ResponsePojo.class).jobsIds.single();
    }

    @Override
    public Option<OnetimeJob> getOnetimeJob(FullJobId id) {
        throw new NotImplementedException("not implemented");
    }

    @Override
    public boolean isJobActive(OnetimeTask task) {
        throw new NotImplementedException("not implemented");
    }

    @Override
    public ListF<? extends OnetimeTask> getActiveTasks(ListF<? extends OnetimeTask> tasks) {
        throw new NotImplementedException("not implemented");
    }

    @Override
    public ListF<OnetimeJob> getActiveJobs(TaskId taskId, SqlLimits limits) {
        return getJobs(taskId, JobStatus.READY, limits);
    }

    @Override
    public ListF<OnetimeJob> getFailedJobs(TaskId taskId, SqlLimits limits) {
        return getJobs(taskId, JobStatus.FAILED, limits);
    }

    @Override
    public ListF<OnetimeJob> getJobsByGroup(String group, SqlLimits limits) {
        throw new NotImplementedException("not implemented");
    }

    private ListF<OnetimeJob> getJobs(TaskId taskId, JobStatus status, SqlLimits limits) {
        ListF<String> params = Cf.list("task", taskId.getId(), "status", status.name().toLowerCase());

        if (!limits.isAll()) {
            params = params.plus(Cf.list("offset", "" + limits.getFirst(), "limit", "" + limits.getCount()));
        }
        String url = actionUrl("get-tasks", params.toArray(String.class));

        return executeAndParse(new HttpGet(url), PgTasksActionContainer.TasksPojo.class).tasks;
    }

    private String actionUrl(String action, String... parameters) {
        return UrlUtils.addParameters(
                "http://" + hostPort.toSerializedString() + "/tasks/" + action, Tuple2List.fromPairs(parameters));
    }

    private <T> HttpEntity serializeRequestEntity(T bendable) {
        BenderSerializer<T> serializer = PgTasksActionConfigurator.benderMapper.createSerializer(
                ClassX.wrap(bendable.getClass()).<T>uncheckedCast().getClazz());

        return new ByteArrayEntity(serializer.serializeJson(bendable), ContentType.APPLICATION_JSON);
    }

    private <T> T executeAndParse(HttpUriRequest request, Class<T> responseClass) {
        try {
            return httpClient.execute(request, parseResponseHandler(request.getURI().getPath(), responseClass));
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    private <T> ResponseHandler<T> parseResponseHandler(String operation, Class<T> clazz) {
        return response -> {
            Response resp;
            Option<T> result;
            try {
                InputStreamSource content = InputStreamSourceUtils.wrap(response.getEntity().getContent());
                resp = PgTasksActionConfigurator.benderMapper.parseJson(Response.class, content);
                result = resp.result.map(o -> PgTasksActionConfigurator.benderMapper.createParser(clazz).parseJson(o));

            } catch (Exception e) {
                throw new RuntimeException(""
                        + "Got " + response.getStatusLine().getStatusCode()
                        + " and failed to parse response of " + operation + ": " + e, e);
            }
            if (resp.error.isPresent()) {
                Option<String> host = resp.invocationInfo.flatMapO(i -> i.hostname);
                Option<String> reqId = resp.invocationInfo.flatMapO(i -> i.reqId);

                Option<String> name = resp.error.get().name.filterNot(String::isEmpty);
                Option<String> message = resp.error.get().message.filterNot(String::isEmpty);

                throw new RuntimeException(""
                        + "Invocation of " + operation + " failed"
                        + host.map(h -> " at " + h + reqId.map(rid -> " <" + rid + ">").mkString("")).mkString("")
                        + name.plus(message).mkString(": ", ": ", ""));
            }
            return result.get();
        };
    }

    @BenderBindAllFields
    private static class Response {
        public Option<InvocationInfo> invocationInfo;
        public Option<BenderJsonNode> result;
        public Option<ResponseError> error;
    }

    @BenderBindAllFields
    private static class ResponseError {
        public Option<String> name;
        public Option<String> message;
    }

    @BenderBindAllFields
    private static class InvocationInfo {
        @BenderPart(name = "req-id", strictName = true)
        public Option<String> reqId;
        public Option<String> hostname;
    }

    @Override
    public void close() throws IOException {
        HttpClientUtils.closeQuietly(httpClient);
    }
}
