package ru.yandex.chemodan.bazinga.http;

import java.util.UUID;

import javax.annotation.PostConstruct;

import org.joda.time.Instant;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTaskRegistry;
import ru.yandex.chemodan.core.worker.tasks.ShellOnetime;
import ru.yandex.chemodan.core.worker.tasks.ShellOnetimeParameters;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobId;
import ru.yandex.commune.bazinga.impl.JobInfoValue;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.ip.IpPortUtils;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

/**
 * @author dbrylev
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = PgHttpBazingaTaskManagerTest.ContextConfiguration.class)
public class PgHttpBazingaTaskManagerTest {

    private static final PgBazingaStorage storage = Mockito.mock(PgBazingaStorage.class);

    @Autowired
    private PgHttpBazingaTaskManager manager;

    @Test
    public void schedule() {
        FullJobId jobId = new FullJobId(new TaskId("task"), new JobId(UUID.randomUUID()));

        ArgumentCaptor<OnetimeJob> jobCaptor = ArgumentCaptor.forClass(OnetimeJob.class);
        Mockito.when(storage.addOnetimeJob(jobCaptor.capture(), Mockito.any())).thenReturn(jobId);

        Instant date = MoscowTime.instant(2016, 2, 24, 23, 40);
        ShellOnetime job = new ShellOnetime(new ShellOnetimeParameters(Cf.list("ls"), Option.empty()));

        Assert.equals(jobId, manager.schedule(job, Option.of("activeUid"), date, 77));
        Assert.equals(date, jobCaptor.getValue().getScheduleTime());
        Assert.equals(77, jobCaptor.getValue().getPriority());
        Assert.some("activeUid", jobCaptor.getValue().getActiveUniqueIdentifier());

        Mockito.when(storage.addOnetimeJob(Mockito.any(), Mockito.any())).thenThrow(new IllegalArgumentException());
        Assert.assertThrows(() -> manager.schedule(job), RuntimeException.class);
    }

    @Test
    public void getTasks() {
        TaskId taskId = new TaskId("task");

        OnetimeJob job = new OnetimeJob(
                new FullJobId(taskId, new JobId(UUID.randomUUID())),
                Option.of(Instant.now()),
                Instant.now(),
                Option.empty(), Option.of("activeUid"), "{}",
                JobInfoValue.ready(),
                Option.empty(), Cf.set(), Option.empty(),
                27, Option.empty());

        ArgumentCaptor<SqlLimits> limitsCaptor = ArgumentCaptor.forClass(SqlLimits.class);
        ArgumentCaptor<ListF<JobStatus>> statusesCaptor = ArgumentCaptor.forClass(
                ClassX.wrap(ListF.class).<ListF<JobStatus>>uncheckedCast().getClazz());

        Mockito.when(storage.findOnetimeJobsByTaskIdAndStatuses(
                Mockito.any(), statusesCaptor.capture(), limitsCaptor.capture())).thenReturn(Cf.list(job));

        ListF<OnetimeJob> jobs = manager.getActiveJobs(taskId, SqlLimits.all());
        Assert.equals(job.toString(), jobs.single().toString());
        Assert.equals(SqlLimits.all(), limitsCaptor.getValue());

        manager.getFailedJobs(taskId, SqlLimits.first(10));
        Assert.equals(SqlLimits.first(10), limitsCaptor.getValue());
        Assert.equals(Cf.list(JobStatus.FAILED), statusesCaptor.getValue());

        manager.getActiveJobs(taskId, SqlLimits.range(10, 10));
        Assert.equals(SqlLimits.range(10, 10), limitsCaptor.getValue());
        Assert.equals(Cf.list(JobStatus.READY), statusesCaptor.getValue());
    }

    @Configuration
    public static class ContextConfiguration {

        private static int port = IpPortUtils.getFreeLocalPort().getPort();

        @Bean
        public PgBazingaStorage pgBazingaStorage() {
            return storage;
        }

        @Bean
        public ActionApp actionApp(PgBazingaStorage storage) {
            return PgTasksActionConfigurator.configure(
                    new ApplicationInfo("", ""), Cf.list(new PgTasksActionContainer(
                            new ConfigurableOnetimeTaskRegistry(), storage, new PgBazingaTaskManager(storage), 1)));
        }

        @Bean
        public SingleWarJetty singleWarJetty(ActionApp actionApp) {
            SingleWarJetty jetty = new SingleWarJetty();
            jetty.setHttpPort(port);
            jetty.addServletMapping("/tasks/*", actionApp.createServlet());
            return jetty;
        }

        @Bean
        public PgHttpBazingaTaskManager pgHttpTaskManager() {
            return new PgHttpBazingaTaskManager(
                    new HostPort("localhost", port),
                    ApacheHttpClientUtils.singleConnectionClient(new Timeout(10000, 10000)));
        }

        @Autowired
        private SingleWarJetty jetty;

        @PostConstruct
        public void start() {
            jetty.start();
        }
    }
}
