package ru.yandex.chemodan.bazinga.http;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.OnetimeTaskUtils;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTask;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTaskRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.thread.factory.IncrementThreadFactory;

/**
 * @author yashunsky
 */

@ActionContainer
public class PgTasksActionContainer {

    private final ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry;
    private final PgBazingaStorage storage;
    private final PgBazingaTaskManager taskManager;
    private final ExecutorService executor;

    public PgTasksActionContainer(ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry,
            PgBazingaStorage storage,
            PgBazingaTaskManager taskManager,
            int poolSize)
    {
        this.configurableOnetimeTaskRegistry = configurableOnetimeTaskRegistry;
        this.storage = storage;
        this.taskManager = taskManager;
        this.executor = Executors.newFixedThreadPool(poolSize, new IncrementThreadFactory("PgTasksActionContainer-pool-"));
    }

    @Path("/get-tasks-count")
    public TasksCountPojo getTasksCount(
            @RequestParam("task") String taskId,
            @RequestParam("status") ListF<JobStatus> statuses)
    {
        return new TasksCountPojo(
                statuses.map(status -> storage.findOnetimeJobCount(new TaskId(taskId), status)).sum(Cf.Integer)
        );
    }

    @Path("/get-tasks")
    public TasksPojo getTasks(
            @RequestParam("task") String taskId,
            @RequestParam("status") ListF<JobStatus> statuses,
            @RequestParam("offset") Option<Integer> offset,
            @RequestParam("limit") Option<Integer> limit)
    {
        return new TasksPojo(
                storage.findOnetimeJobsByTaskIdAndStatuses(new TaskId(taskId), statuses,
                        limit.map(l -> SqlLimits.range(offset.getOrElse(0), l)).getOrElse(SqlLimits.all()))
        );
    }

    private FullJobId scheduleTask(AddTasksRequest taskParams) {

        OnetimeTask task;

        if (configurableOnetimeTaskRegistry.getTask(taskParams.taskId).isPresent()) {
            ConfigurableOnetimeTask.Parameters parameters =
                    BazingaBender.mapper.createParser(ConfigurableOnetimeTask.Parameters.class)
                            .parseJson(taskParams.parameters);
            task = configurableOnetimeTaskRegistry.getTask(taskParams.taskId).get()
                    .makeCopy(parameters);
        } else {
            task = OnetimeTaskUtils.makeOnetimeTask(new TaskId(taskParams.taskId), taskParams.parameters);
        }
        return taskManager.schedule(task,
                taskParams.activeUniqueIdentifier,
                taskParams.scheduleTime.getOrElse(Instant.now()),
                taskParams.priority.getOrElse(task.priority()));
    }

    @Path(value = "/add-tasks", methods = HttpMethod.POST)
    public ResponsePojo addTasks(@BoundByBender AddTasksRequestList request) {

        ListF<CompletableFuture<FullJobId>> futures = request.tasks.map(
                taskParams -> CompletableFutures.supplyAsync(() -> scheduleTask(taskParams), executor)
        );

        return new ResponsePojo(CompletableFutures.join(CompletableFutures.allOf(futures)));
    }

    @BenderBindAllFields
    public static class AddTasksRequestList {
        public final ListF<AddTasksRequest> tasks;

        public AddTasksRequestList(
                ListF<AddTasksRequest> tasks)
        {
            this.tasks = tasks;
        }
    }

    @BenderBindAllFields
    public static class AddTasksRequest {
        public final String taskId;
        public final String parameters;
        public final Option<Integer> priority;
        public final Option<Instant> scheduleTime;
        public final Option<String> activeUniqueIdentifier;

        public AddTasksRequest(String taskId, String parameters, Option<Integer> priority,
                Option<Instant> scheduleTime, Option<String> activeUniqueIdentifier)
        {
            this.taskId = taskId;
            this.parameters = parameters;
            this.priority = priority;
            this.scheduleTime = scheduleTime;
            this.activeUniqueIdentifier = activeUniqueIdentifier;
        }
    }

    @BenderBindAllFields
    @ActionResultPojo
    public static class TasksCountPojo {
        public final int count;

        public TasksCountPojo(int count) {
            this.count = count;
        }
    }

    @BenderBindAllFields
    @ActionResultPojo
    public static class TasksPojo {
        public final ListF<OnetimeJob> tasks;

        public TasksPojo(ListF<OnetimeJob> tasks) {
            this.tasks = tasks;
        }
    }

    @BenderBindAllFields
    @ActionResultPojo
    public static class ResponsePojo {
        public final ListF<FullJobId> jobsIds;

        public ResponsePojo(ListF<FullJobId> jobsIds) {
            this.jobsIds = jobsIds;
        }
    }
}
