package ru.yandex.chemodan.bazinga.monica;

import java.util.concurrent.atomic.AtomicLong;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.ValueMap;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class BazingaTaskQueueMetrics extends DelayingWorkerServiceBeanSupport implements MonicaContainer {
    private final PgBazingaStorage bazingaStorage;
    private final BazingaWorkerApp bazingaWorkerApp;

    @MonicaMetric
    private final ValueMap<Long> countByTaskAndStatus = new ValueMap<>(Long.class, 1000);

    public BazingaTaskQueueMetrics(PgBazingaStorage bazingaStorage, BazingaWorkerApp bazingaWorkerApp) {
        this.bazingaStorage = bazingaStorage;
        this.bazingaWorkerApp = bazingaWorkerApp;
    }

    @Override
    protected Duration defaultDelay() {
        return Duration.standardSeconds(30);
    }

    @Override
    protected void execute() throws Exception {
        AtomicLong totalQueue = new AtomicLong(0);
        bazingaWorkerApp.getWorkerTaskRegistry().getOnetimeTasks().forEach(ot -> {
            MapF<JobStatus, Long> counts = Cf.hashMap();
            for (JobStatus status : JobStatus.values()) {
                long count = bazingaStorage.findOnetimeJobCounters(ot.id(), Cf.list(status));

                countByTaskAndStatus.set(count, new MetricName(ot.id().toString(), status.toString() + "_max"));
                counts.put(status, count);
            }

            long queue = counts.getOrElse(JobStatus.READY, 0L) + counts.getOrElse(JobStatus.RUNNING, 0L) + counts.getOrElse(JobStatus.STARTING, 0L);
            totalQueue.addAndGet(queue);
            countByTaskAndStatus.set(queue, new MetricName(ot.id().toString(), "queue_max"));
        });

        countByTaskAndStatus.set(totalQueue.get(), new MetricName("total_queue_max"));
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "bazinga",
                new MetricName("bazinga", "task_queue"),
                "Bazinga job queue"
        );
    }
}
