package ru.yandex.chemodan.notifier;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class AddLentaNotificationBlockTask
        extends YcridOnetimeTaskSupport<AddLentaNotificationBlockTask.Parameters>
{
    private static final Logger logger = LoggerFactory.getLogger(AddLentaNotificationBlockTask.class);

    public AddLentaNotificationBlockTask() {
        super(Parameters.class);
    }

    public AddLentaNotificationBlockTask(Parameters parameters) {
        super(parameters);
    }

    public AddLentaNotificationBlockTask(DataApiUserId uid, String blockId, Instant deadline) {
        this(new Parameters(uid, blockId, YandexCloudRequestIdHolder.getO(), Option.empty(), Option.of(deadline)));
    }

    public AddLentaNotificationBlockTask(DataApiUserId uid, String blockId, boolean sendPushNow, Instant deadline) {
        this(new Parameters(uid, blockId, YandexCloudRequestIdHolder.getO(),
                Option.of(sendPushNow), Option.of(deadline)));
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {
        throw new UnsupportedOperationException("not supported");
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        if (parameters.deadline.exists(deadline -> deadline.isBefore(Instant.now()))) {
            logger.info("Skip adding block {} for {} because deadline {} passed",
                    parameters.activeUid.blockId, parameters.activeUid.uid, parameters.deadline.get());
            return;
        }
        super.execute(parameters, context);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Override
    public TaskQueueName queueName() {
        return NotificationTaskQueueName.NOTIFIER_REGULAR;
    }

    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return ActiveUidParams.Converter.class;
    }

    @BenderBindAllFields
    public static class Parameters extends YcridTaskParameters {
        @BenderFlatten
        public final ActiveUidParams activeUid;

        public final Option<Boolean> sendImmediately;
        public final Option<Instant> deadline;

        public Parameters(DataApiUserId uid, String blockId,
                Option<String> ycrid, Option<Boolean> sendImmediately, Option<Instant> deadline)
        {
            super(ycrid);
            this.sendImmediately = sendImmediately;
            this.activeUid = new ActiveUidParams(uid, blockId);
            this.deadline = deadline;
        }
    }

    @BenderBindAllFields
    @Data
    public static class ActiveUidParams {
        public final DataApiUserId uid;
        public final String blockId;

        public ActiveUidParams(DataApiUserId uid, String blockId) {
            this.uid = uid;
            this.blockId = blockId;
        }

        public static class Converter implements ActiveUniqueIdentifierConverter<Parameters, ActiveUidParams> {
            @Override
            public Class<ActiveUidParams> getActiveUniqueIdentifierClass() {
                return ActiveUidParams.class;
            }

            @Override
            public ActiveUidParams convert(Parameters parameters) {
                return parameters.activeUid;
            }
        }
    }
}
