package ru.yandex.chemodan.queller.admin;

import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Supplier;

import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.HttpClientUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.celery.worker.WorkerState;
import ru.yandex.chemodan.queller.celery.worker.WorkerStateProviderHolder;
import ru.yandex.chemodan.queller.rabbit.RabbitConnectionHealth;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.admin.web.AdminBender;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZMapperSource;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.InstantAsMillisMarshaller;
import ru.yandex.misc.bender.custom.InstantAsMillisUnmarshaller;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.io.http.apache.v4.ReadBytesResponseHandler;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.version.SimpleAppName;

/**
 * @author yashunsky
 */
@ActionContainer
public class CeleryMonitorAdminPage implements Closeable, ZMapperSource {

    private static final BenderMapper benderMapper = new BenderMapper(new BenderConfiguration(
            AdminBender.SETTINGS,
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(Instant.class, new InstantAsMillisMarshaller(), new InstantAsMillisUnmarshaller())
                    .build()));

    private final WorkerStateProviderHolder monitor;
    private final RabbitPool rabbitPool;

    private final AliveAppsHolder aliveAppsHolder;
    private final SimpleAppName appName;

    private final Supplier<ListF<String>> conductorHostsSupplier;

    private final ExecutorService executor = Executors.newFixedThreadPool(50);
    private final HttpClient httpClient;

    public CeleryMonitorAdminPage(
            WorkerStateProviderHolder monitor, RabbitPool rabbitPool,
            AliveAppsHolder aliveAppsHolder, SimpleAppName appName,
            Supplier<ListF<String>> conductorHostsSupplier,
            HttpClient httpClient)
    {
        this.monitor = monitor;
        this.rabbitPool = rabbitPool;
        this.aliveAppsHolder = aliveAppsHolder;
        this.appName = appName;
        this.conductorHostsSupplier = conductorHostsSupplier;
        this.httpClient = httpClient;
    }

    @ZAction(defaultAction = true)
    @Path("/celery-monitor")
    public MonitorPojo index() {
        String localHost = HostnameUtils.localHostname();
        MonitorPojo localState = local();

        ListF<String> aliveHosts = aliveAppsHolder.aliveApps().filterMap(
                app -> Option.when(app.getServiceName().equals(appName.serviceName())
                        && app.getAppName().equals(appName.appName())
                        && !app.getHostname().equals(localHost), app.getHostname()));

        Function<String, MonitorPojo> retrieveF = MasterSlaveContextHolder.withStandardThreadLocalsF(host -> {
            try {
                String url = pageUrl(host) + "/local.json";
                return benderMapper.parseJson(MonitorPojo.class,
                        httpClient.execute(new HttpGet(url), new ReadBytesResponseHandler()));

            } catch (Exception e) {
                return new MonitorPojo(new AppRabbitConnectionHealths(host, localPageUrl(host), e));
            }
        });
        ListF<MonitorPojo> aliveStates = CompletableFutures.join(CompletableFutures.allOf(
                aliveHosts.map(host -> CompletableFuture.supplyAsync(() -> retrieveF.apply(host), executor))));

        Tuple2List<String, MonitorPojo> hostsStates = aliveHosts.zip(aliveStates).plus1(localHost, localState);

        hostsStates = hostsStates.plus(conductorHostsSupplier.get()
                .filterNot(hostsStates.get1()::containsTs)
                .zipWith(h -> new MonitorPojo(new AppRabbitConnectionHealths(h, localPageUrl(h), "Not alive"))));

        ListF<MonitorPojo> states = hostsStates.sortedBy1().get2();

        return new MonitorPojo(
                Option.of(states.exists(s -> s.hasWorkers.isSome(true))),
                states.flatMap(p -> p.workers).stableUniqueBy(w -> w.workerId),
                states.flatMap(p -> p.appRabbits));
    }

    @ZAction(file = "CeleryMonitorAdminPage-index.xsl")
    @Path("/celery-monitor/local")
    public MonitorPojo local() {
        String localHost = HostnameUtils.localHostname();

        return new MonitorPojo(
                Option.of(monitor.isSet()),
                monitor.getO().map(m -> m.getWorkersState().values()).getOrElse(Cf.list()),
                Cf.list(new AppRabbitConnectionHealths(
                        localHost, localPageUrl(localHost), rabbitPool.getHealth().values())));
    }

    private String localPageUrl(String host) {
        return pageUrl(host) + "/local";
    }

    private String pageUrl(String host) {
        return "http://" + appName.serviceName() + "-" + appName.appName() + "." + host + ":81/z/celery-monitor";
    }

    @XmlRootElement(name = "content")
    @BenderBindAllFields
    public static class MonitorPojo {
        public final Option<Boolean> hasWorkers;
        public final ListF<WorkerState> workers;
        public final ListF<AppRabbitConnectionHealths> appRabbits;

        public MonitorPojo(
                Option<Boolean> hasWorkers,
                CollectionF<WorkerState> workers,
                CollectionF<AppRabbitConnectionHealths> appRabbits)
        {
            this.hasWorkers = hasWorkers;
            this.workers = workers.toList().sorted(WorkerId.comparator.compose(w -> WorkerId.parse(w.workerId)));
            this.appRabbits = appRabbits.toList().sortedBy(h -> h.host);
        }

        public MonitorPojo(AppRabbitConnectionHealths appRabbits) {
            this(Option.empty(), Cf.list(), Cf.list(appRabbits));
        }
    }

    @BenderBindAllFields
    public static class AppRabbitConnectionHealths {
        public final String host;
        public final String url;
        public final Option<String> error;
        public final ListF<RabbitConnectionHealth> rabbits;

        public AppRabbitConnectionHealths(
                String host, String url, Option<String> error, ListF<RabbitConnectionHealth> rabbits)
        {
            this.host = host;
            this.url = url;
            this.error = error;
            this.rabbits = rabbits.sortedBy(r -> r.host);
        }

        public AppRabbitConnectionHealths(String host, String url, CollectionF<RabbitConnectionHealth> rabbits) {
            this(host, url, Option.empty(), rabbits.toList());
        }

        public AppRabbitConnectionHealths(String host, String url, String error) {
            this(host, url, Option.of(error), Cf.list());
        }

        public AppRabbitConnectionHealths(String host, String url, Throwable error) {
            this(host, url, ExceptionUtils.getAllMessages(error));
        }
    }

    @Override
    public BenderMapper mapper() {
        return benderMapper;
    }

    @Override
    public void close() throws IOException {
        HttpClientUtils.closeQuietly(httpClient);
        executor.shutdownNow();
    }
}
